package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.ShipFactory;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import com.google.common.base.Optional;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.Random;
import java.util.Set;

/**
 * State of all the sea pirates.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 1, 2013
 *
 */
@LazySingleton
public class SeaPiratesState {
	@Autowired
	private Random rnd;
	@Autowired
	private ShipFactory shipFactory;
	@Autowired
	private IMap map;

	/** List of the non free pirates */
	private Multimap<IPlayer, ISeaPirate> nonFreePirates = ArrayListMultimap.create();
	/** List of all free pirates */
	private Set<ISeaPirate> freePirates = Sets.newHashSet();

	/*
	 * TODO use pirate names
	 */
	private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
	private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");
	RandomNameLoader shipLoader = new RandomNameLoader("shipnames.properties");

	@PostConstruct
	public void init() {
		String name = firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName();
		IShip ship = createInitialPiratesShip();
		ISeaPirate pirate = createNewPirate(name, ship);
		Optional<IPlayer> absent = Optional.absent();
		add(pirate, absent);

	}

	/**
	 * Create a ship for the pirate
	 * @return
	 */
	private IShip createInitialPiratesShip() {
		// TODO aho Jan 29, 2013: upgrade the ship, add weapons and sailors
		int shipChoice = rnd.nextInt(3);
		IShip ship = null;
		switch (shipChoice) {
		case 0:
			ship = shipFactory.createSnaikka(shipLoader.getRandomName(), 0); // size of no consequence
			break;
		case 1:
			ship = shipFactory.createCrayer(shipLoader.getRandomName(), 0);
			break;
		case 2:
			ship = shipFactory.createHolk(shipLoader.getRandomName(), 0);
			break;
		default:
			throw new RuntimeException("Should never get here");
		}
		return ship;
	}

	/**
	 * Add a pirate. The pirate may be a private pirate belonging only to one player.
	 * @param pirate
	 * @param player
	 */
	public void add(ISeaPirate pirate, Optional<IPlayer> player) {
		if (player.isPresent()) {
			nonFreePirates.put(player.get(), pirate);
		} else {
			freePirates.add(pirate);
		}
	}
	/**
	 * A non free pirate becomes free.
	 * @param pirate
	 */
	public void free(IPlayer player, ISeaPirate pirate) {
		nonFreePirates.remove(player, pirate);
		freePirates.add(pirate);
	}
	/**
	 * Retrieve immutable list of non free pirates for the player
	 * @param player
	 * @return
	 */
	public List<ISeaPirate> getNonFreePirates(IPlayer player) {
		return ImmutableList.copyOf(nonFreePirates.get(player));
	}
	/**
	 * Retrieve immutable list of free pirates.
	 * @return
	 */
	public Set<ISeaPirate> getFreePirates(){
		return ImmutableSet.copyOf(freePirates);
	}
	/**
	 * Create an new pirate anywhere on the sea
	 * @param name
	 * @param ship
	 * @return
	 */
	public ISeaPirate createNewPirate(String name, IShip ship) {
		SeaPirate pirate = new SeaPirate();
		pirate.setName(name);
		pirate.setShip(ship);
		pirate.moveTo(new Point2D(50,50)); // TODO aho Jan 29, 2013: randomize and set it anywhere on sea
		return pirate;
	}
	/**
	 * Create a pirate near a certain point
	 * @param name
	 * @param ship
	 * @param near
	 * @return
	 */
	public ISeaPirate createNewPirate(String name, IShip ship, Point2D near) {
		SeaPirate pirate = (SeaPirate) createNewPirate(name, ship);
		double newX = Math.min(Math.max(rnd.nextInt(100)-50+near.getX(),0), map.getDimension().getWidth());
        double newY = Math.min(Math.max(rnd.nextInt(100)-50+near.getY(),0), map.getDimension().getHeight());
		Point2D p = new Point2D(newX, newY);
		pirate.moveTo(p);
		return pirate;
	}
	/**
	 * Find a successful free pirate
	 * @return
	 */
	public ISeaPirate findSuccessfulPirate() {
		int maxKill = 0;
		ISeaPirate successful = null;
		for (ISeaPirate pirate : freePirates) {
			if (pirate.getNumberOfKills()>=maxKill) {
				successful = pirate;
				maxKill = pirate.getNumberOfKills();
			}
		}
		return successful;
	}
}
