package ch.sahits.game.openpatrician.model.personal.impl;

import ch.sahits.game.event.data.PeriodicalTimeWeekUpdate;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.LoanGivingEvent;
import ch.sahits.game.openpatrician.model.event.RemitLoanEvent;
import ch.sahits.game.openpatrician.model.factory.PeriodicalTimeUpdateFactory;
import ch.sahits.game.openpatrician.model.people.ReputationCalculator;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.time.EUpdateIntervalRegistration;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

/**
 * Implementation of the reputation.
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 23, 2012
 *
 */
@Prototype
public class Reputation implements IReputation {

	private int reputationLastWeek = 0;
	private int reputationCurrentWeek = 0;

	private final ICity city;
	private final IPlayer player;
	@Autowired
	private ReputationCalculator repCalc;
	@Autowired
	private PeriodicalTimeUpdateFactory timeUpdateFactory;
	@Autowired
	@Qualifier("serverClientEventBus")
	private AsyncEventBus clientServerEventBus;

	public Reputation(ICity city, IPlayer player) {
		super();
		this.city = city;
		this.player = player;
	}
	private void init() {
		clientServerEventBus.register(this);
		// This will create a bean and register with the event bus. So we do not need to keep a reference.
		timeUpdateFactory.createPeriodicalUpdater(EUpdateIntervalRegistration.WEEK);
	}

	@Override
	public int getReputationDifference() {
		return reputationCurrentWeek-reputationLastWeek;
	}

	/**
	 * Handling the daily update events.
	 * @param event
	 */
	@Subscribe
	public void handleWeeklyUpdate(PeriodicalTimeWeekUpdate event) {
		reputationLastWeek=reputationCurrentWeek;
		reputationCurrentWeek=repCalc.calculateWareRepputation(city, player);
		// TODO add other aspects
	}

	@Override
	public int getPopularity() {
		return reputationCurrentWeek;
	}

	/**
	 * When a loan is given this improves the reputation.
	 * @param event
	 */
	@Subscribe
	public void handleLoanGiving(LoanGivingEvent event) {
		if (event.getDebt().getCreditor().equals(player) && event.getCity().equals(city)) {
			reputationCurrentWeek += event.getDebt().getAmount() / 1000 * 10;
		}
	}
	@Subscribe
	public void handleRemitLoan(RemitLoanEvent event) {
		if (event.getDebt().getCreditor().equals(player) && event.getCity().equals(city)) {
			reputationCurrentWeek += 500;
		}
	}

}
