package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.model.BaseAmountable;
import javafx.beans.binding.NumberBinding;
import javafx.beans.binding.When;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
import ch.sahits.game.openpatrician.model.IAmountable;
import ch.sahits.game.openpatrician.model.product.IWare;
/**
 * This class defines the amounted price of an {@link IWare} object.
 * Each object in the real world has hits unique price. Since no instances of the
 * objects are used but only one reference the price information gets lost. The
 * amountable price tracks the average price of all amounted objects
 * @param <T> {@link IAmountable} implementation that is is collected hereby called item
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 20, 2011
 * @param <T>
 *
 */
public class AmountablePrice<T extends IAmountable> extends BaseAmountable<T> {
	/** Total price of all amounted items */
	private final DoubleProperty sum = new SimpleDoubleProperty(this, "sum", 0);
	/**
	 * Retrieve the average price of one item
	 * @return
	 */
	public int getAVGPrice(){
		return (int)Math.rint(sum.doubleValue()/amount.get());
	}
	
	public NumberBinding avgPriceProperty() {
		return new When(sum.isEqualTo(0)
				        .or(amount.isEqualTo(0)))
		          .then(0)
		          .otherwise(sum.divide(amount));
	}
	
	
	/**
	 * Add a number items
	 * @param amount number of the items to be added
	 * @param avgPrice average price of one item that is added
	 */
	public void add(int amount, int avgPrice){
		if (amount<0){
			amount = Math.abs(amount);
		}
		int current = this.amount.get();
		this.amount.set(current+amount);
		long sum = this.sum.longValue();
		this.sum.set(sum + amount*avgPrice);
	}
	/**
	 * Remove a number of items
	 * @param amount of items to be removed
	 */
	public void remove (int amount){
		if (amount<0){
			amount = Math.abs(amount);
		}
		if (amount==Integer.MIN_VALUE){
			amount = Integer.MAX_VALUE;
		}
		if (this.amount.get()<amount){
			this.amount.set(0);
			sum.set(0);
		} else {
			long sum = this.sum.longValue();
			this.sum.set(sum - amount*getAVGPrice()); // sum must be updated first
			int current = this.amount.get();
			this.amount.set(current - amount);
		}
		if (this.amount.get()==0){
			sum.set(0);
		}
	}
	
	@Override
	public void reset() {
		this.sum.set(0);
		super.reset();
	}

	public String toString(){
		return amount.get()+"@"+getAVGPrice();
	}
}
