package ch.sahits.game.openpatrician.model.ship;

import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import com.google.common.base.Optional;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.geometry.Point2D;

import java.util.Set;

/**
 * Model of a ship
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jun 15, 2011
 *
 */
public interface IShip {
	// TODO separate methods in ship and single ship as not all methods make sense for a convoy
	/**
	 * Total capacity of the ship in barrels
	 * @return
	 */
	public int getSize();
	/**
	 * Available capacity of the ship, this represent the free storage in barrels
	 * @return
	 */
	public int getCapacity();
	/**
	 * Check if the ship can be upgraded
	 * @return
	 */
	public boolean isUpgradable();
	/**
	 * Upgrade the ship if it is upgradable
	 */
	public void upgrade();
	/**
	 * Retrieve the name of the of the ship
	 * @return
	 */
	public String getName();
	/**
	 * Set the ships name
	 * @param name
	 */
	public void setName(String name);
	/**
	 * Retrieve a list of all loaded wares
	 * @return
	 */
	public Set<IWare> getLoadedWares();
	/**
	 * Load the ware in the specified amount onto the ship.
	 * The amount uses the size specified by the ware
	 * @param ware to be loaded
	 * @param amount of items the ware
	 * @return amount actually loaded
	 * @param avgPrice average price of one item of the ware
	 */
	public int load(IWare ware, int amount, int avgPrice);
	/**
	 * Unload the specified quantity of the ware.
	 * The amount uses the size specified by the ware
	 * @param ware to be unloaded
	 * @param amount amount of items of ware to unload
	 * @return amount of the ware unloaded
	 */
	public int unload(IWare ware, int amount);
	/**
	 * Retrieve the damage of the ship in percentage [0,100]
	 * @return
	 */
	public int getDamage();
	/**
	 * Damage the ship
	 * @param damage
	 */
	public void damage(Object damage); // TODO create a damage object
	/**
	 * Repair the damage
	 */
	public void repair();
	/**
	 * Retrieve the reduction of the load capacity based on the upgrade level.
	 * The reduction is in barrel that reduces the initial size
	 * @return reduction in barrels
	 */
	public int getUpgradeSpaceReduction();
	/**
	 * Retrieve the amount of load in barrels that is loaded in actual wares (no wappen or crew)
	 * @return amount of loaded wares in barrels
	 */
	@Deprecated
	public int getLoad();
	/**
	 * Retrieve the amount of load in barrels that is loaded in actual wares (no wappen or crew)
	 * @return amount of loaded wares in barrels
	 */
	public IntegerBinding getLoadBinding();
	/**
	 * Retrieve the amount of ware loaded
	 * @param ware
	 * @return
	 */
	public AmountablePrice<IWare> getWare(IWare ware);
	/**
	 * check if weapons are on board
	 * @return
	 */
	public boolean hasWeapons();
	/**
	 * Retrieve the current value of the ship without its cargo
	 * @return
	 */
	public int getValue();
	/**
	 * Retrieve the number of sailors on the ship.
	 * @return
	 */
	int getNumberOfSailors();
	/**
	 * Set the number of sailors.
	 * @param nbSailors .
	 */
	void setNumberOfSailors(int nbSailors);

    IntegerProperty numberOfSailorsProperty();
	/**
	 * Retrieve the minimum of sailors needed to operate the ship.
	 * @return
	 */
	int getMinNumberOfSailors();
	/**
	 * Retrieve the maximum of sailors that can be put on the ship.
	 * This amount depends on the ship type as well as the amount of load.
	 * @return
	 */
	int getMaxNumberOfSailors();
	/**
	 * Retrieve the captain of the ship.
	 * There might not be a captain present.
	 * @return
	 */
	Optional<ICaptain> getCaptian();
	/**
	 * Set the captain on the ship
	 * @param captain
	 */
	void setCaptain(ICaptain captain);
	/**
	 * Move weapons in or out.
	 * @param weaponType SWORD | BOW | CROSSBOW | MUSKET
	 * @param amount
	 */
	void move(IWeapon weaponType, int amount);
	/**
	 * Retrieve the amount of weapons on the ship
	 * @param weaponType SWORD | BOW | CROSSBOW | MUSKET
	 * @return amount available on the ship
	 */
	int getWeaponAmount(IWeapon weaponType);
	/**
	 * Move a passenger to the ship. The ship may only contain one passanger.
	 * @param passanger
	 */
	void addPassenger(IPerson passanger);
	/**
	 * Passenger leave ship.
	 */
	void leavePassanger();

    /**
     * Retrieve the passenger on the ship.
     * @return optional wrapped around the passenger.
     */
    Optional<IPerson> getPassenger();
	/**
	 * Retrieve the location of the ship.
	 * @return
	 */
	Point2D getLocation();

    /**
     * Set the location of the ship.
     * @param location of the ship
      */
    void setLocation(Point2D location);
	/**
	 * Retrieve the owner of the ship. The owner cannot be retrieved if the pirate flag is set.
	 * @return
	 */
	IPlayer getOwner();
	/**
	 * Flag indicating if the pirate flag is set. This means the owner can not be retrieved.
	 * @return
	 */
	boolean isPiratingShip();
	/**
	 * Retrieve the type of the ship.
	 * @return
	 */
	EShipType getShipType();
	/**
	 * Define the owner of the ship
	 * @param player
	 */
	void setOwner(IPlayer player);
	/**
	 * Retrieve the upgrade level of the ship.
	 * @return
	 */
	EShipUpgrade getShipUpgradeLevel();
	
	IShipWeaponsLocation getShipWeaponsLocation();

	/**
	 * Indicate if the ship is available. A ship may be unavailable during repair or upgrade.
	 * @return
	 */
	boolean isAvailable();

	/**
	 * Set the available flag on the ship.
	 * @param available
	 */
	void setAvailable(boolean available);

	int getOccupiedSpace();
	void setOccupiedSpace(int space);
}
