package ch.sahits.game.openpatrician.model.time;

import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.event.data.ClockTickIntervalChange;
import ch.sahits.game.event.data.PeriodicalTimeDayUpdate;
import ch.sahits.game.event.data.PeriodicalTimeMonthUpdate;
import ch.sahits.game.event.data.PeriodicalTimeWeekUpdate;
import ch.sahits.game.event.data.PeriodicalTimeYearUpdate;
import ch.sahits.game.openpatrician.model.Date;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;

/**
 * The periodical time updater encapsules clock ticks of instances that are interested in
 * notification but only on a periodical level. The period may vary but the implementation is
 * always the same. The periods as defined in {@link ch.sahits.game.openpatrician.model.time.EUpdateIntervalRegistration} are possible,
 * where as a year is composed by 365 days and a month has always 30 days. So these periods are
 * not really exact.
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 22, 2012
 *
 */
public class PeriodicalTimeUpdaterV2 {
	/** Period of the clock ticking */
	private int tickInterval;
    @Autowired
	private Date date;
	/** Period that the listener should be notified */
	private int updateInterval;
    private final EUpdateIntervalRegistration intervall;
    /** Time passed since the start of the period */
    private int timePassed = 0;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;

    public PeriodicalTimeUpdaterV2(EUpdateIntervalRegistration intervall){
        Preconditions.checkArgument(intervall != EUpdateIntervalRegistration.DAILY, "This updater does not handle daily events");
        this.intervall = intervall;
        System.out.println("Create periodical updater for intervall "+intervall);
    }
    @PostConstruct
    public void initializeEventBus() {
        tickInterval = date.getTickUpdate();
        final DateTime cal = date.getCurrentDate();
        switch (intervall) {
            case YEAR:
                updateInterval = 24*60*365;
                timePassed = (cal.getYear()-1)*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case MONTH:
                updateInterval = 24*60*30;
                timePassed = (cal.getDayOfMonth()-1)*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case WEEK:
                updateInterval = 7*24*60;
                timePassed = cal.getDayOfWeek()*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case END_OF_DAY:
                updateInterval = 24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            // TODO add last of month
            default:
                throw new IllegalArgumentException("Unknown update interval: "+intervall);
        }
        clientServerEventBus.register(this);
    }


	@Subscribe
	public void updateTick(ClockTickIntervalChange event ) {
		tickInterval = event.getInterval();
	}


    @Subscribe
    public void handleClockTick(ClockTick event) {
        timePassed += tickInterval;
        if (timePassed>updateInterval){
            switch (intervall) {
                case YEAR:
                    clientServerEventBus.post(new PeriodicalTimeYearUpdate());
                    break;
                case MONTH:
                    clientServerEventBus.post(new PeriodicalTimeMonthUpdate());
                    break;
                case WEEK:
                    clientServerEventBus.post(new PeriodicalTimeWeekUpdate());
                    break;
                case END_OF_DAY:
                    clientServerEventBus.post(new PeriodicalTimeDayUpdate());
                    break;
                // TODO add last of month
                default:
                    throw new IllegalArgumentException("Unknown update interval: "+intervall);                }
            timePassed=0;
        }

    }

    public void unregisterFromEventBus() {
        clientServerEventBus.unregister(this);
    }

}
