package ch.sahits.game.openpatrician.model.city;

import ch.sahits.game.openpatrician.data.map.Cities;
import ch.sahits.game.openpatrician.data.map.City;
import ch.sahits.game.openpatrician.data.map.Map;
import lombok.Getter;
import lombok.Setter;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.oxm.Unmarshaller;
import org.springframework.stereotype.Component;

import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Factory class for the cities of the game.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
@Component
public class CityFactory { // todo: andi 12/20/14: This class should only be instantiated on the server
    private static final Logger logger = Logger.getLogger(CityFactory.class);

    @Autowired
    @Setter
    @Qualifier("jaxb2MapMarshaller")
    private Unmarshaller unmarshaller;
    @Autowired
    private ApplicationContext context;

	/**
	 * Cache storing the already created cities to  avoid duplicate creation
	 */
	private static ConcurrentHashMap<String,ICity> createdCityCache = new ConcurrentHashMap<>();
    @Getter
    private Map map;

    private List<City> getCities(String mapName) {
        try {
            map = (Map) unmarshaller.unmarshal(getSourceFromFile(mapName));
            Cities cities = map.getCities();
            return cities.getCity();
        } catch (IOException e) {
            logger.error("Could not load cities from file "+mapName, e);
            return newArrayList();
        }
    }

    private Source getSourceFromFile(String fileName) {
        final InputStream resourceAsStream = getClass().getResourceAsStream(fileName);
        return new StreamSource(resourceAsStream);
    }

    /**
     * Initialize all cities defined in <code>mapName</code>
     * @param mapName name of the XML file
     * @throws IOException invalid XML file
     */
    public void initializeCityCache(String mapName) throws IOException {
        List<City>  cities = getCities(mapName);
        for (City city : cities) {
            createdCityCache.put(city.getName(), createCity(city));
        }
    }

    private ICity createCity(City city) {
        return (ICity) context.getBean("cityBean", new Object[]{city});
    }

    /**
     * Retrieve all cities from the internal cache.
     * @return Collection of cities.
     */
    public Collection<ICity> getCities() {
        return createdCityCache.values();
    }

    /**
	 * Create a standard city based on its name
	 * @param cityName
	 * @return Instance of the city
	 * @throws IOException Error reading the city properties
	 */
	public ICity createCityByName(String cityName) throws IOException{
		return createdCityCache.get(cityName.toUpperCase());
	}
}
