package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.model.GameFactory;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.util.GenericPair;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;

/**
 * State common to all cities.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 1, 2013
 *
 */
@Lazy
@Component
public class CitiesState {
	@Autowired
	private IMap map;
	@Autowired
	private Random rnd;
	@Autowired
	private GameFactory gameFactory;


	/** List of all the cities coupled with their state to be handeld by this engine */
	/* Use of  ConcurrentHashMap because later in the game cities might be added and we want
	 * to prevent ConcurrentModificationExceptions if we are currently iterating when the
	 * new city is added.
	 * Use map to retrieve city.
	 */
	private ConcurrentHashMap<ICity, CityState> cities = new ConcurrentHashMap<ICity, CityState>();
		@PostConstruct
		public void init() {
			for (ICity city : map.getCities()) {
				CityState cityState = gameFactory.getCityState(city);
				cities.put(city, cityState);

			}
		}

	/**
	 * Find a possible city that has needed wares. Repeated calls to this method are not garanteed to
	 * return the same city.
	 * @return
	 */
	public Optional<GenericPair<ICity, IWare>> findCityWithMostNeededWare() {
		List<GenericPair<ICity, IWare>> all = new ArrayList<GenericPair<ICity,IWare>>();
		for (CityState state : cities.values()) {
			Optional<IWare> ware = state.findMostNeededWare();
			if (ware.isPresent()) {
				all.add(new GenericPair<ICity, IWare>(state.getCity(), ware.get()));
			}
		}
		if (all.isEmpty()) {
			return Optional.absent();
		} else {
			return Optional.of(all.get(rnd.nextInt(all.size())));
		}
	}

	/**
	 * Find a possible city that has needed wares. Repeated calls to this method are not garanteed to
	 * return the same city.
	 * @return
	 */
	public Optional<GenericPair<ICity, IWare>> findCityWithSurplusWare() {
		List<GenericPair<ICity, IWare>> all = new ArrayList<GenericPair<ICity,IWare>>();
		for (CityState state : cities.values()) {
			Optional<IWare> ware = state.findWareWithMostSurplus();
			if (ware.isPresent()) {
				all.add(new GenericPair<ICity, IWare>(state.getCity(), ware.get()));
			}
		}
		if (all.isEmpty()) {
			return Optional.absent();
		} else {
			return Optional.of(all.get(rnd.nextInt(all.size())));
		}
	}

	public List<CityState> getCityEngineStates(){
		return ImmutableList.copyOf(cities.values());
	}
}
