package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.city.CityProduction;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.factory.StateFactory;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import com.google.common.base.Optional;
import lombok.Getter;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;

/**
 * This class encapsulates state information used by CityEngine about one city.
 * This class i package private since it should not be used by any other class than CityEngine.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 29, 2011
 *
 */
@Prototype
public class CityState {
	@Autowired
	private Random rnd;
	@Value("${ware.surplus.treshold}")
	private double surplusThreshold;

	/** Reference to the city model that is driven by this engine */
    @Getter
	private ICity city;
	/** Map of consumed wares, since only whole units can be consumed we need to store the fractions */
	private Map<IWare, Double> consumed = new HashMap<IWare, Double>();
	/** Map of produced wares, since only whole units can be consumed we need to store the fractions */
	private Map<IWare, Double> produced = new HashMap<IWare, Double>();
	/** Mock up to simulate the city production */
	@Autowired
	private CityProduction production;
	/** Tavern state for this city */
    @Getter
	private TavernState tavernState;
	/** Ship yard state fos the city */
    @Getter
	private IShipyard shipyardState;
	/** Provider for the consume numbers */
	@Autowired
	private PopulationConsume consume;
	@Autowired
	private StateFactory stateFactory;

	public CityState(ICity city) {
		this.city = city;
	}
	@PostConstruct
	private void initCity() {
		shipyardState = stateFactory.createShipYard(city);
		EWare[] wares = EWare.values();
		for (EWare ware : wares) {
			consumed.put(ware, 0.0);
			produced.put(ware, 0.0);
		}
		tavernState = stateFactory.createTavernState(city);
		city.setCityState(this);
	}
	/**
	 * compute the wares consumed in the last 12 hours and remove them from the city
	 * This method is package private for testing purposes
	 */
	public void consumeWares() {
		EWare[] wares = EWare.values();
		EPopulationClass[] popClasses = EPopulationClass.values();
		for (EWare ware : wares) {
			double amount = consumed.get(ware);
			for (EPopulationClass popClass : popClasses) {
				amount += consume.getNeed(ware, popClass, city.getPopulation(popClass));
			}
			consumed.put(ware, amount);// does this produce ConcurrentModExc?
		}
		for (Entry<IWare, Double> entry : consumed.entrySet()) {
			if (entry.getValue()>=1){
				int whole = (int) Math.rint(Math.floor(entry.getValue()));
				consumed.put(entry.getKey(), entry.getValue()-whole); // does this produce ConcurrentModExc?
				city.move(entry.getKey(), -whole,null); // the city consumes the ware
			}
		}
	}

	/**
	 * Compute the wares produced in the last 12 hours and add the whole quatities to the city.
	 * This method is package private for testing purposes
	 */
	public void produceWares() {
		IWare[] wares = city.getEffectiveProduction();
		for (IWare ware : wares) {
			double amount = produced.get(ware);
			amount += production.getEfficientProduction(ware)/(7.0*2); // the amount is for a whole week
			produced.put(ware, amount);// does this produce ConcurrentModExc?
		}
		wares = city.getIneffectiveProduction();
		for (IWare ware : wares) {
			double amount = produced.get(ware);
			amount += production.getInefficientProduction(ware)/(7.0*2); // the amount is for a whole week
			produced.put(ware, amount);// does this produce ConcurrentModExc?
		}
		for (Entry<IWare, Double> entry : produced.entrySet()) {
			if (entry.getValue()>=1){
				int whole = (int) Math.rint(Math.floor(entry.getValue()));
				produced.put(entry.getKey(), entry.getValue()-whole); // does this produce ConcurrentModExc?
				city.move(entry.getKey(), whole,null); // The city produces the ware
			}
		}
	}
	/**
	 * Find the ware which is consumed the most and which is missing for the longest time.
	 * @return
	 */
	public Optional<IWare> findMostNeededWare() {
		Map<IWare, DateTime> missing = city.getMissingWares();
		DateTime earliest = null;
		IWare result = null;
		for (Entry<IWare, DateTime> entry : missing.entrySet()) {
			if (earliest == null ||
					entry.getValue().isBefore(earliest)) {
				earliest = entry.getValue();
				result = entry.getKey();
			}
		}
		return Optional.fromNullable(result);
	}
	/**
	 * Find the ware that the city sells near the minimal price
	 * @return
	 */
	public Optional<IWare> findWareWithMostSurplus() {
		ArrayList<IWare> surplus = new ArrayList<IWare>();
		for (IWare ware : EWare.values()) {
			AmountablePrice<IWare> amount = city.getWare(ware);
			double actualPrice = amount.getAVGPrice();
			int minPrice = ware.getMinValueSell();
			double val = minPrice/actualPrice - 1;
			if (val<=surplusThreshold) {
				surplus.add(ware);
			}
		}
		if (surplus.isEmpty()) {
			return Optional.absent();
		} else {
			return Optional.of(surplus.get(rnd.nextInt(surplus.size())));
		}
	}


}
