package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipDueDate;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.ShipFactory;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import lombok.Getter;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;

import static com.google.common.collect.Maps.newHashMap;
/**
 * Implementation of the shipyard model.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Mar 9, 2013
 *
 */
@Prototype
public class ShipyardState implements IShipyard {

    // todo: andi 23/02/14: check if the dependency on the client event bus works and think if that is a good solution?

	private static final int BUILD_TIME_SNAIKKA = 30;
	private static final int BUILD_TIME_CRAYER = 40;
	private static final int BUILD_TIME_COG = 60;
	private static final int BUILD_TIME_HOLK = 80;
	private static final int REPAIR_TIME_SNAIKKA = 1;
	private static final int REPAIR_TIME_CRAYER = 2;
	private static final int REPAIR_TIME_COG = 4;
	private static final int REPAIR_TIME_HOLK = 5;

	private static final int MATERIAL_UNIT_COST = 30;
	private static final int BASE_DAY_SALERY = 5;

	private static final Optional<TimedTask> ABSENT_TIMED_TASK = Optional.absent();

	private final Map<IShipDueDate, Optional<TimedTask>> repairList = newHashMap();
	private final Map<IShipDueDate, Optional<TimedTask>> buildList = newHashMap();
	private final Map<IShipDueDate, Optional<TimedTask>> refitList = newHashMap();

	/** Value between 0 and 1 detailing the experience of the ship yard crew (the higher the better) */
	private double experience;
	@Autowired
	private Date date;
	@Autowired
	private Random rnd;
    @Getter
	private final ICity city;
	@Autowired
	private TimedUpdatableTaskList taskList;
	@Autowired
	private ShipFactory shipUtility;
	@Autowired
	private IMap map;

	public ShipyardState(ICity city) {
		this.city = city;
	}

	@PostConstruct
	public void initializeExperience() {
		int mapWidth = (int) map.getDimension().getWidth();
		double eastFactor = city.getCoordinates().getX()/mapWidth/0.5;
		double startYearPart = Math.max((date.getStartYear()-1430)/100,0);
		experience = Math.max(startYearPart-eastFactor,0);
		Preconditions.checkArgument(experience>=0, "Experiance must be positive");
	}

	@Override
	public List<IShipDueDate> getShipBuildingList() {
		cleanup();
		return ImmutableList.copyOf(buildList.keySet());
	}


    @Override
    public void addShipBuildingOrder(IShipDueDate dueDate, Optional<TimedTask> task) {
        buildList.put(dueDate, task);
    }

    @Override
	public void addShipBuildingOrder(IShipDueDate dueDate) {
		buildList.put(dueDate, ABSENT_TIMED_TASK);
	}

	@Override
	public void cancelShipBuildingOrder(DateTime finish) {
		cleanup();
		for (Iterator<IShipDueDate> iterator = buildList.keySet().iterator(); iterator.hasNext();) {
			IShipDueDate dueDate = iterator.next();
			if (dueDate.getDueDate().equals(finish)) {
				Optional<TimedTask> task = buildList.get(dueDate);
				if (task.isPresent()) {
					taskList.remove(task.get());
				}
				iterator.remove();
				break;
			}
		}
	}

	@Override
	public List<IShipDueDate> getShipRepairList() {
		cleanup();
		return ImmutableList.copyOf(repairList.keySet());
	}


    @Override
    public void addShipRepairOrder(IShipDueDate dueDate, Optional<TimedTask> task) {
        repairList.put(dueDate, task);
        experience = Math.max(experience+0.005, 1);
    }

    @Override
	public int calculateRepairMaterialCosts(IShip ship, int damage) {
		double shipSize = getShipSize(ship);
		int materialCosts = (int)(damage*10*shipSize*MATERIAL_UNIT_COST);
		return materialCosts;
	}

	private double getShipSize(IShip ship) {
		double shipSize = 0;
		switch (ship.getShipType()) {
		case SNAIKKA:
			shipSize = 1;
			break;
		case CRAYER:
			shipSize = 1.5;
			break;
		case COG:
			shipSize = 2.5;
			break;
		case HOLK:
			shipSize = 5;
			break;
		}
		return shipSize;
	}
	@Override
	public int calculateRepairCosts(int repairTime, int materialCosts) {
		return repairTime*calculateSaleryPerDay()+materialCosts;
	}
	@Override
	public void addShipToRepairList(IShipDueDate dueDate) {
		repairList.put(dueDate, ABSENT_TIMED_TASK);
	}

	/**
	 * Salery is defined by the experiance
	 * @return
	 */
	private int calculateSaleryPerDay() {
		return (int) (BASE_DAY_SALERY*(1+experience*10));
	}
	@Override
	public void cancelRepair(IShip ship) {
		cleanup();
		for (Iterator<IShipDueDate> iterator = repairList.keySet().iterator(); iterator.hasNext();) {
			IShipDueDate dueDate = iterator.next();
			if (dueDate.getShip().equals(ship)) {
				Optional<TimedTask> task = repairList.get(dueDate);
				if (task.isPresent()) {
					taskList.remove(task.get());
				}
				iterator.remove();
				ship.setAvailable(true);
				break;
			}
		}
	}


    @Override
    public void addShipRefitOrder(IShipDueDate dueDate, Optional<TimedTask> task) {
        refitList.put(dueDate, task);
    }

    @Override
	public int calculateRefitTime(EShipType type) {
		return calculateBuildTime(type)/10;
	}
	@Override
	public void addToRefitList(IShipDueDate dueDate) {
		refitList.put(dueDate, ABSENT_TIMED_TASK);
	}

	@Override
	public DateTime getBuildCompleteDate(EShipType type) {
		cleanup();
		DateTime lastOrderFinished = date.getCurrentDate();
		for (IShipDueDate dueDate : buildList.keySet()) {
			if (dueDate.getDueDate().isAfter(lastOrderFinished)) {
				lastOrderFinished = dueDate.getDueDate();
			}
		}
		int buildTime = calculateBuildTime(type);
		return lastOrderFinished.plusDays(buildTime);
	}
	/**
	 * Calculate the build time for the ship of type.
	 * @param type
	 * @return
	 */
	@Override
	public int calculateBuildTime(EShipType type) {
		int baseBuildTime = 0;
		switch (type) {
		case SNAIKKA:
			baseBuildTime = BUILD_TIME_SNAIKKA;
			break;
		case CRAYER:
			baseBuildTime = BUILD_TIME_CRAYER;
			break;
		case COG:
			baseBuildTime = BUILD_TIME_COG;
			break;
		case HOLK:
			baseBuildTime = BUILD_TIME_HOLK;
			break;
		}
		return (int) Math.rint(baseBuildTime+(1-experience)*baseBuildTime);
	}
	/**
	 * Calculate the repair time for the ship of type.
	 * @param type
	 * @return
	 */
	@Override
	public int calculateRepairTime(EShipType type) {
		int baseRepairTime = 0;
		switch (type) {
		case SNAIKKA:
			baseRepairTime = REPAIR_TIME_SNAIKKA;
			break;
		case CRAYER:
			baseRepairTime = REPAIR_TIME_CRAYER;
			break;
		case COG:
			baseRepairTime = REPAIR_TIME_COG;
			break;
		case HOLK:
			baseRepairTime = REPAIR_TIME_HOLK;
			break;
		}
		return (int) Math.rint(baseRepairTime+(1-experience)*baseRepairTime);
	}	/**
	 * Remove entries that lie in the past. This are orders that are fullfiled and
	 * can be removed.
	 */
	public void cleanup() {
		DateTime now = date.getCurrentDate();
		for (Iterator<IShipDueDate> iterator = buildList.keySet().iterator(); iterator.hasNext();) {
			IShipDueDate dueDate = iterator.next();
			if (dueDate.getDueDate().isBefore(now)) {
				iterator.remove();
			}
		}
		for (Iterator<IShipDueDate> iterator = repairList.keySet().iterator(); iterator.hasNext();) {
			IShipDueDate dueDate = iterator.next();
			if (dueDate.getDueDate().isBefore(now)) {
				iterator.remove();
			}
		}
		for (Iterator<IShipDueDate> iterator = refitList.keySet().iterator(); iterator.hasNext();) {
			IShipDueDate dueDate = iterator.next();
			if (dueDate.getDueDate().isBefore(now)) {
				iterator.remove();
			}
		}
	}
	@Override
	public int calculateRefitCosts(EShipType type, int levels) {
		return calculateRefitTime(type)*levels*calculateSaleryPerDay();
	}
	@Override
	public int calculateConstructionCosts(EShipType type) {
		return calculateBuildTime(type)*calculateSaleryPerDay();
	}

	@Override
	public EShipType[] getBuildableShipTypes() {
		// Cog 1420 + experiance > 0.2
		// Holk 1450 + expertice > 0.4
		if (date.getCurrentDate().getYear() > 1450 && experience > 0.4) {
			return new EShipType[]{EShipType.SNAIKKA, EShipType.CRAYER, EShipType.COG, EShipType.HOLK};
		} else if (date.getCurrentDate().getYear() > 1420 && experience > 0.2) {
			return new EShipType[]{EShipType.SNAIKKA, EShipType.CRAYER, EShipType.COG};
		} else {
			return new EShipType[]{EShipType.SNAIKKA, EShipType.CRAYER};
		}
	}

}
