package ch.sahits.game.openpatrician.model.event;

import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.openpatrician.model.Date;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.joda.time.DateTime;
import org.joda.time.DateTimeComparator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.util.Comparator;
import java.util.Iterator;
import java.util.TreeSet;

/**
 * List of tasks that are to be executed at a certain time.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 28, 2013
 *
 */
// This class is not annotated as service/component to be able to have different lists under different names
public class TimedUpdatableTaskList  {
	@Autowired
	private Date date;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;

	private final TreeSet<TimedTask> sortedEvents;

	/**
	 * Register as Clock update listener.
	 */
	@PostConstruct
	public void registerWithClock() {
        clientServerEventBus.register(this);
	}

	public TimedUpdatableTaskList(){
		Comparator<TimedTask> comparator = new Comparator<TimedTask>() {
			private final DateTimeComparator dateTimeComparator = DateTimeComparator.getInstance();
			@Override
			public int compare(TimedTask task1, TimedTask task2) {
				return dateTimeComparator.compare(task1.getExecutionTime(), task2.getExecutionTime());
			}
		};
		sortedEvents = new TreeSet<TimedTask>(comparator);
	}
	/**
	 * Add a new task.
	 * @param task
	 */
	public void add(TimedTask task) {
		sortedEvents.add(task);
	}
	/**
	 * Cancel a timed task prematurely.
	 * @param task
	 */
	public void remove(TimedTask task) {
		sortedEvents.remove(task);
	}
    @Subscribe
    public void handleClockTick(ClockTick event) {
        DateTime now = date.getCurrentDate();
        for (Iterator<TimedTask> iterator = sortedEvents.iterator(); iterator.hasNext();) {
            TimedTask task = iterator.next();
            Preconditions.checkNotNull(task.getExecutionTime(), "The exectution time of the task may not be null");
            if (task.getExecutionTime().isBefore(now)) {
                task.run();
                iterator.remove();
            }
            if (task.getExecutionTime().isAfter(now)) {
                break; // List is sorted
            }
        }
    }

	/**
	 * Retrieve the first task. Only used for tests.
	 * @return
	 */
	TimedTask getFirst() {
		return sortedEvents.first();
	}
}
