package ch.sahits.game.openpatrician.model.product;

import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.When;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import ch.sahits.game.openpatrician.model.city.ECityState;
import ch.sahits.game.openpatrician.model.city.IPopulationStructure;
/**
 * The most simple form of the price calculation algorithm.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 27, 2011
 *
 */
final class LinearPriceCalculation implements IPriceCalculation {

	/**
	 * Compute the price for one item of the ware. Though the price is only handled in further
	 * computation as integer value, this method returns its value as double so this condition can be
	 * tested:<br>
	 * The higher the available amount (but still below the saturation value) the smaller (strictly) the returned
	 * value.<br>
	 * For different starting values:<br>
	 * If  the starting value b is below starting point a the whole curve of b is below (not necessary strictly)
	 * the curve of a. The curve is continues between 0 (inclusive) and infinity.<br>
	 * This implementation uses a linear function between the max value at 0 and the min value at saturation.
	 * This however means that the function is not continuous at the point of the saturation.
	 * @param min the lowest possible price (0 exclusive)
	 * @param max the highest possible price
	 * @param available amount of wares that are available in the market
	 * @param saturation amount of items at which the market is saturated (no further price change
	 * @param productionRate amount the city can produce within a week
	 * @param pop population structure that may have an influence on the calculated price
	 * @param state state of the city
	 * @return price for one item sold
	 */	@Override
	public double computePrice(int min, int max, int available, int saturation,
			int productionRate, IPopulationStructure pop, ECityState state) {
		 if (available==0) return max;
		 if (available>=saturation) return min;
		double slope = -(max-min)/(1.0*saturation);
		return available*slope+max;
	}
	 // // TODO aho Nov 30, 2013: remove
	public IntegerBinding computePrice(ReadOnlyIntegerProperty min,
			ReadOnlyIntegerProperty max, IntegerProperty available,
			ReadOnlyIntegerProperty saturation, int productionRate,
			IPopulationStructure pop, ECityState state) {
		return (IntegerBinding) new When(available.isEqualTo(0))
		.then(max)
		.otherwise(new When(available.greaterThanOrEqualTo(saturation))
					.then(min)
					.otherwise(available
							     .multiply(max
							    		    .subtract(min)
							    		    .multiply(-1)
							    		   .divide(saturation))
							     .add(max))
		);
	}

}
