package ch.sahits.game.openpatrician.model.ship;

import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.product.EWare;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PropertiesLoaderUtils;

import java.io.IOException;
import java.util.Properties;

/**
 * This factory class creates ship instances.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 19, 2011
 *
 */
@LazySingleton
public class ShipFactory { // todo: andi 12/20/14: This class should only be instantiated on the server
	private  final Logger logger = Logger.getLogger(getClass());
	@Autowired
	private Date date;
	@Autowired
	private IMap map;
	@Autowired
	private ApplicationContext context;

	private static final int SNAIKKA_START_CAPACITY = 150;
	private static final int SNAIKKA_MAX_CAPACITY = 250;
	private static final int CRAYER_START_CAPACITY = 280;
	private static final int CRAYER_MAX_CAPACITY = 350;
	private static final int COG_START_CAPACITY = 450;
	private static final int COG_MAX_CAPACITY = 580;
	private static final int HOLK_START_CAPACITY = 550;
	private static final int HOLK_MAX_CAPACITY = 700;


	private Properties getHolkDefaultProperties() throws IOException {
		Resource resource = new ClassPathResource("/ship_holk.properties");
		return PropertiesLoaderUtils.loadProperties(resource);
	}

	private Properties getCogDefaultProperties() throws IOException {
		Resource resource = new ClassPathResource("/ship_cog.properties");
		return PropertiesLoaderUtils.loadProperties(resource);
	}

	private Properties getCrayerDefaultProperties() throws IOException {
		Resource resource = new ClassPathResource("/ship_crayer.properties");
		return PropertiesLoaderUtils.loadProperties(resource);
	}

	private Properties getSnikkaDefaultProperties() throws IOException {
		Resource resource = new ClassPathResource("/ship_schnikka.properties");
		return PropertiesLoaderUtils.loadProperties(resource);
	}


	/**
	 * Create a a crayer with the given name. The crayer is not upgraded
	 * @param name of the ship
	 * @param value size of the ship
	 * @return
	 */
	public ICrayer createCrayer(String name,int value){
		Properties crayer = null;
		try {
			crayer = getCrayerDefaultProperties();
			crayer.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (ICrayer) context.getBean("crayer", new Object[]{name, crayer});
	}
	/**
	 * Create a crayer with the given name. The crayer is upgraded to the specified level
	 * @param name of the ship
	 * @param value size of the ship
	 * @param upgradeLevel
	 * @return
	 */
	public ICrayer createCrayer(String name, EShipUpgrade upgradeLevel,int value){
		Properties crayer = null;
		try {
			crayer = getCrayerDefaultProperties();
			crayer.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (ICrayer) context.getBean("crayer", new Object[]{upgradeLevel, name, crayer});
	}
	/**
	 * Create a a snaikka with the given name. The snaikka is not upgraded
	 * @param name of the ship
	 * @param value size of the ship
	 * @return
	 */
	public ISnaikka createSnaikka(String name,int value){
		Properties snikka = null;
		try {
			snikka = getSnikkaDefaultProperties();
			snikka.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (ISnaikka) context.getBean("snaikka", new Object[]{name, snikka});
	}
	/**
	 * Create a snaikka with the given name. The snaikka is upgraded to the specified level
	 * @param name of the ship
	 * @param value size of the ship
	 * @param upgradeLevel
	 * @return
	 */
	public ISnaikka createSnaikka(String name, EShipUpgrade upgradeLevel,int value){
		Properties snikka = null;
		try {
			snikka = getSnikkaDefaultProperties();
			snikka.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (ISnaikka) context.getBean("snaikka", new Object[]{upgradeLevel, name, snikka});
	}
	/**
	 * Create a a cog with the given name. The cog is not upgraded
	 * @param name of the ship
	 * @param value size of the ship
	 * @return
	 */
	public ICog createCog(String name,int value){
		Properties cog = null;
		try {
			cog = getCogDefaultProperties();
			cog.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (ICog) context.getBean("cog", new Object[]{name, cog});
	}
	/**
	 * Create a cog with the given name. The cog is upgraded to the specified level
	 * @param name of the ship
	 * @param value size of the ship
	 * @param upgradeLevel
	 * @return
	 */
	public ICog createCog(String name, EShipUpgrade upgradeLevel,int value){
		Properties cog = null;
		try {
			cog = getCogDefaultProperties();
			cog.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (ICog) context.getBean("cog", new Object[]{upgradeLevel, name, cog});
	}
	/**
	 * Create a a holk with the given name. The holk is not upgraded
	 * @param name of the ship
	 * @param value size of the ship
	 * @return
	 */
	public IHolk createHolk(String name,int value){
		Properties holk = null;
		try {
			holk = getHolkDefaultProperties();
			holk.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (IHolk) context.getBean("holk", new Object[]{name, holk});
	}
	/**
	 * Create a holk with the given name. The holk is upgraded to the specified level
	 * @param name of the ship
	 * @param value size of the ship
	 * @param upgradeLevel
	 * @return
	 */
	public IHolk createHolk(String name, EShipUpgrade upgradeLevel,int value){
		Properties holk = null;
		try {
			holk = getHolkDefaultProperties();
			holk.setProperty("size", String.valueOf(value));
		} catch (IOException e) {
			logger.error("Failed to initialize ship properties", e);
		}
		return (IHolk) context.getBean("holk", new Object[]{upgradeLevel, name, holk});
	}
	/**
	 * Calulate the ships type's capacity based on the current year and the
	 * eastwards degrading.
	 * @param type ship type
	 * @param x coordinate on where to build it.
	 * @return
	 */
	public int calculateInitialCapacity(EShipType type, double x) {
		double startYearPart = Math.max((date.getCurrentDate().getYear()-1430)/100,0);
		int mapWidth = (int) map.getDimension().getWidth();
		double eastFactor = x/mapWidth/0.5;
		double factor = Math.max(startYearPart-eastFactor,0);
		int sizeDiff;
		switch (type) {
		case SNAIKKA:
			sizeDiff = SNAIKKA_MAX_CAPACITY-SNAIKKA_START_CAPACITY;
			return (int) (SNAIKKA_START_CAPACITY+Math.rint(factor*sizeDiff));
		case CRAYER:
			sizeDiff = CRAYER_MAX_CAPACITY-CRAYER_START_CAPACITY;
			return (int) (CRAYER_START_CAPACITY+Math.rint(factor*sizeDiff));
		case COG:
			sizeDiff = COG_MAX_CAPACITY-COG_START_CAPACITY;
			return (int) (COG_START_CAPACITY+Math.rint(factor*sizeDiff));
		case HOLK:
			sizeDiff = HOLK_MAX_CAPACITY-HOLK_START_CAPACITY;
			return (int) (HOLK_START_CAPACITY+Math.rint(factor*sizeDiff));
		}
		return 0;
	}
	/**
	 * Retrieve the amount that is needed for the ship to upgrade to the next level
	 * @param type ship type
	 * @param ware that is required
	 * @return ware specific amount of the ware.
	 */
	public int getUpgradeAmount(EShipType type,EWare ware) {
		int typeAddition = 0;
		switch (type) {
		case SNAIKKA:
			typeAddition = 0;
			break;
		case CRAYER:
			typeAddition = 1;
			break;
		case COG:
			typeAddition = 2;
			break;
		case HOLK:
			typeAddition = 3;
			break;
		}
		switch (ware) {
		case WOOD:
			return 2 + typeAddition;
		case PITCH:
		case IRON:
			return 1 + typeAddition;
		default:
			return 0;
		}
	}
	/**
	 * Retrieve the amount that is needed for the ship to upgrade to the next level
	 * @param type ship type
	 * @param ware that is required
	 * @return ware specific amount of the ware.
	 */
	public int getConstructionAmount(EShipType type,EWare ware) {
		switch (type) {
		case SNAIKKA:
			switch (ware) {
			case WOOD:
				return 11;
			case CLOTH:
				return 3;
			case IRON:
				return 3;
			case HEMP:
				return 3;
			case PITCH:
				return 20;
			default:
				throw new IllegalArgumentException("Not defined: "+type+" "+ware);
			}
		case CRAYER:
			switch (ware) {
			case WOOD:
				return 16;
			case CLOTH:
				return 5;
			case IRON:
				return 5;
			case HEMP:
				return 5;
			case PITCH:
				return 30;
			default:
				throw new IllegalArgumentException("Not defined: "+type+" "+ware);
			}
		case COG:
			switch (ware) {
			case WOOD:
				return 22;
			case CLOTH:
				return 3;
			case IRON:
				return 4;
			case HEMP:
				return 4;
			case PITCH:
				return 40;
			default:
				throw new IllegalArgumentException("Not defined: "+type+" "+ware);
			}
		case HOLK:
			switch (ware) {
			case WOOD:
				return 36;
			case CLOTH:
				return 10;
			case IRON:
				return 10;
			case HEMP:
				return 8;
			case PITCH:
				return 50;
			default:
				throw new IllegalArgumentException("Not defined: "+type+" "+ware);
			}
		}
		return 0;
	}
	/**
	 * Get the minimum of sailors per ship type.
	 * @param type of the ship
	 * @return number of sailors
	 */
	public int getMinimalSailors(EShipType type) {
		switch (type) {
		case SNAIKKA:
			return 5;
		case CRAYER:
			return 8;
		case COG:
			return 10;
		case HOLK:
			return 15;			
		}
		return 0;
	}

	/**
	 * Get the speed of the ship in knots.
	 * @param type of the ship
	 * @return speed
	 */
	public int getShipSpeed(EShipType type) {
		switch (type) {
		case SNAIKKA:
			return 5;
		case CRAYER:
			return 6;
		case COG:
			return 4;
		case HOLK:
			return 5;			
		}
		return 0;
	}

	
}
