package ch.sahits.game.openpatrician.model.util;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import javafx.geometry.Point2D;

import org.springframework.beans.factory.annotation.Autowired;

import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.city.ICity;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Range;

/**
 * Utility class to privide services around the cities.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 27, 2013
 *
 */
@LazySingleton
public class CityUtilities {
	@Autowired
	private IMap map;
	@Autowired
	private Random rnd;

	private final Range<Double> range = Range.openClosed(0.0, 1.0);

	/**
	 * Find a city nearby. The search is limited by a factor (0,1]. Based on
	 * the <code>max</code> dimension of the map the radius is calculated with this
	 * factor.
	 * Repeated calls with the same arguments are not guaranteed to deliver the same result
	 * @param startCity city where to start
	 * @param radiusFactor factor for the radius calculation
	 * @param max maximal dimension of the map
	 * @param exclude potentially empty list of cities, that should be excluded from the search
	 * @return Optional city within the radius.
	 */
	public Optional<ICity> findNearbyCity(ICity startCity, double radiusFactor, int max, List<ICity> exclude) {
		List<ICity> cities = map.getCities();
		Preconditions.checkArgument(range.contains(radiusFactor), "Range factor must be in (0,1]");
		final double radius = max*radiusFactor;
		LinkedList<ICity> copy = new LinkedList<ICity>(cities);
		Collections.shuffle(copy);
		final Point2D p1 = startCity.getCoordinates();
		for (ICity city : copy) {
			if (!exclude.contains(city) && !city.equals(startCity)) {
				Point2D p2 = city.getCoordinates();
				double dist = calculateDistance(p1, p2);
				if (dist<=radius) {
					return Optional.of(city);
				}
			}
		}
		return Optional.absent();
	}
	/**
	 * Find a nearby city. If the radius is chosen to small the radius is increased successively.
	 * This method is guaranteed to deliver a result if there is a city that can be chosen.
	 * @param startCity city where to start
	 * @param radiusFactor factor for the radius calculation
	 * @param max maximal dimension of the map
	 * @param exclude potentially empty list of cities, that should be excluded from the search
	 * @return Optional city within the radius.
	 * @see #findNearbyCity(ICity, double, int, List)
	 */
	public ICity findNearbyCityRepeated(ICity startCity, double radiusFactor, int max, List<ICity> exclude) {
		Preconditions.checkArgument(checkCities(startCity,exclude), "No city is choosable");
		Optional<ICity> result = findNearbyCity(startCity, radiusFactor, max, exclude);
		while (!result.isPresent()) {
			radiusFactor = radiusFactor*1.01;
			result = findNearbyCity(startCity, radiusFactor, max, exclude);
		}
		return result.get();
	}
	/**
	 * Check if there is a city that can be choosen
	 * @param startCity
	 * @param exclude
	 * @return
	 */
	private boolean checkCities(ICity startCity, List<ICity> exclude) {

		List<ICity> cities = map.getCities();
		for (ICity city : cities) {
			if (city.equals(startCity)) {
				continue;
			}
			if (exclude.contains(city)) {
				continue;
			}
			return true;
		}
		return false;
	}
	/**
	 * Find the nearest city to a point
	 * @param p
	 * @return
	 */
	public ICity findNearestCity(Point2D p) {
		List<ICity> cities = map.getCities();
		ICity nearest = cities.get(0);
		double dist = calculateDistance(p, nearest.getCoordinates());
		for (int i=1; i<cities.size(); i++) {
			ICity city = cities.get(i);
			double d = calculateDistance(p, city.getCoordinates());
			if (d<dist) {
				dist = d;
				nearest = city;
			}
		}
		return nearest;
	}
	/**
	 * Find a random city that is not the excluded city.
	 * @param excludedCity city that cannot be chosen.
	 * @return
	 */
	public ICity findRandomCity(ICity excludedCity) {
		List<ICity> cities = Lists.newArrayList(map.getCities());
		cities.remove(excludedCity);
		int index = rnd.nextInt(cities.size());
		return cities.get(index);
	}
	private double calculateDistance(Point2D p1, Point2D p2) {
		double diffX = p1.getX()-p2.getX();
		double diffY = p1.getY()-p2.getY();
		return Math.sqrt(diffX*diffX+diffY*diffY);
	}
}
