package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.collections.NonReplacableMap;
import ch.sahits.game.openpatrician.data.map.Landbridge;
import ch.sahits.game.openpatrician.data.map.Map;
import ch.sahits.game.openpatrician.data.map.Newtown;
import ch.sahits.game.openpatrician.model.city.CityFactory;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IFutureTowns;
import ch.sahits.game.openpatrician.model.city.impl.CityRegistry;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.factory.BuildingFactory;
import ch.sahits.game.openpatrician.model.impl.FutureTowns;
import ch.sahits.game.openpatrician.model.impl.GameMap;
import ch.sahits.game.openpatrician.model.impl.LandBridge;
import ch.sahits.game.openpatrician.model.people.ReputationCalculator;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.impl.PersonalData;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.io.IOException;
import java.util.List;
import java.util.Random;

/**
 * Factory class for game start up. This component is instantiated before the date is set.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 2, 2013
 *
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class GameFactory {   // todo: andi 12/20/14: This class should only be instantiated on the server
    private final Logger logger = LogManager.getLogger(getClass());

	@Autowired
	private Random rnd;
	@Autowired
	private CityFactory cityFactory;
    @Autowired
    private Date date;
	@Autowired
	private BuildingFactory buildingFactory;

	private NonReplacableMap<String, Object> beanMap = new NonReplacableMap<>();


    @Autowired
	private ApplicationContext context;
	@Autowired
	private CityRegistry cityRegistry;
	@Autowired
	private PlayerList players;
    @Autowired
    private AIPlayerList aiPlayers;


    /**
	 * Create a human player instance. Build their trading office in the home town as well.
	 * @param name
	 * @param lastName
	 * @param homeTown
	 * @param male
	 * @return
	 */
	public IPlayer createPlayer(String name, String lastName, ICity homeTown, boolean male, int cash) {
		int age = rnd.nextInt(10)+17;
		PersonalData pd = new PersonalData(name, lastName, male, homeTown, computeRandomBirthDate(age));
		IPlayer player = (IPlayer) context.getBean("player", new Object[]{homeTown, pd,cash,ESocialRank.CHANDLER, this});
		buildingFactory.createTradingOffice(player, homeTown, 0); // initial trading office has no value
		initPlayerInCities(player);
		players.add(player);
		return player;
	}
	/**
	 * Create an artifiacial player instance. Build their trading office in the home town as well.
	 * @param homeTown
	 * @return
	 */
	public IAIPlayer createAIPlayer(ICity homeTown,long cash) {
		int age = rnd.nextInt(15)+17;
		IAIPlayer player = (IAIPlayer) context.getBean("aiPlayer", new Object[]{homeTown,cash,computeRandomBirthDate(age), this});
		buildingFactory.createTradingOffice(player, homeTown, 0); // initial trading office has no value
		initPlayerInCities(player);
		players.add(player);
        aiPlayers.add(player);
		return player;
	}
	/**
	 * Init the player data in all cities
	 * @param player
	 */
	private void initPlayerInCities(IPlayer player) {
        for (ICity city : cityFactory.getCities()) {
            city.moveIn(player);
        }
	}
	/**
	 * Create the  map with all cities defined in its XML definition
	 * @return standard map of the game
	 * @throws IOException Cities properties could not be read
	 */
	public IMap createMap(String mapName) throws IOException{ // todo: andi 06/04/14: pass the XML file name
		cityFactory.initializeCityCache(mapName);
        Map m = cityFactory.getMap();
        // todo: andi 07/04/14: get the details from the cityFactory
		GameMap map = (GameMap) context.getBean(IMap.class);
        final Dimension2D dim = new Dimension2D(m.getDimension().getX(), m.getDimension().getY());
        map.setup(date, cityFactory.getCities(), dim, m.getImageName());
        List<Newtown> newTowns = m.getNewlocations().getNewtown();
        List<IFutureTowns> futures = map.getFutureTowns();
        for (Newtown newTown : newTowns) {
            futures.add(new FutureTowns(newTown.getName(), new Point2D(newTown.getNewlocation().getX(), newTown.getNewlocation().getY())));
        }
        List<Landbridge> bridges = m.getLandbridges().getLandbridge();
        List<ILandBridge> landbridges = map.getLandbridges();
        for (Landbridge bridge : bridges) {
            ICity from = map.findCity(bridge.getFrom());
            ICity to = map.findCity(bridge.getTo());
            if (from == null) {
                 logger.warn("The from location for the land bridge could not be found: "+bridge.getFrom());
            } else if (to == null) {
                logger.warn("The to location for the land bridge could not be found: "+bridge.getTo());
            } else {
                landbridges.add(new LandBridge(from, to));
            }
        }
		context.getBean(StartNewGameBean.class);

		return map;
	}
	public IGame createGame(IMap map, EObjective objective, Difficulty difficulty, EGameSpeed speed, int startYear, boolean singleplayer){
		if (!beanMap.containsKey("game")) {
			beanMap.put("game", context.getBean("game", new Object[]{objective, difficulty, speed, singleplayer}));
		}
		return (IGame) beanMap.get("game");
	}
	public IGame getGame() {
		return (IGame) beanMap.get("game");
	}

	private DateTime computeRandomBirthDate(int age){
		int year = date.getStartYear()-age;
		int month = rnd.nextInt(12)+1;
		int day;
		if (month==2){
			day = rnd.nextInt(28)+1;
		} else if (month==4 || month==6 || month==9 || month==11){
			day = rnd.nextInt(30)+1;
		} else {
			day = rnd.nextInt(31)+1;
		}
		return new DateTime(year, month, day, 0, 0);
	}
	/**
	 * Retrieve the city state for the city. This method makes sure that repeated calls with
	 * the same city argument return the same instance.
	 * @param city
	 * @return
	 */
	public CityState getCityState(ICity city) {
		List<CityState> cityStates = cityRegistry.getCityStates();
		if (cityStates.contains(city)) {
			return cityStates.get(cityStates.indexOf(city));
		} else {
            CityState state = initCityPropertyState(city);
			return state;
		}
	}

    private CityState initCityPropertyState(ICity city) {
        CityState state = (CityState) context.getBean("cityState", new Object[]{city});
		cityRegistry.add(state);
        return state;
    }

    public ReputationCalculator getReputationCalculator() {
		return context.getBean(ReputationCalculator.class);
	}
	public ICompany createCompany(IPlayer owner, ICity homeTown, long cash) {
		return (ICompany) context.getBean("company", new Object[]{owner, homeTown, cash});
	}
}
