package ch.sahits.game.openpatrician.model.building.impl;

import ch.sahits.game.event.data.PeriodicalTimeWeekUpdate;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IAutomatedTrading;
import ch.sahits.game.openpatrician.model.building.ISteward;
import ch.sahits.game.openpatrician.model.building.IStorage;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.factory.BuildingFactory;
import ch.sahits.game.openpatrician.model.factory.PlayerInteractionFactory;
import ch.sahits.game.openpatrician.model.impl.BalanceSheet;
import ch.sahits.game.openpatrician.model.impl.WareHolding;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

/**
 * Implementation of the trading office
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Dec 17, 2011
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class TradingOffice extends WareHolding implements ITradingOffice {
	@Autowired
	@XStreamOmitField
	private BuildingFactory buildingFactory;
	@Autowired
	@XStreamOmitField
	private PlayerInteractionFactory interactionFactory;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;
	/**
	 * Maximal capacity of the trading office in barrels
	 */
	private static final int MAX_CAPACITY = 500;
	/**
	 * Reference to the player
	 */
    @Getter
	private final IPlayer owner;
	/** Reference to the city the trading office is located in */
	@Getter
	private final ICity city;
	/** Value of the building (building costs */
	private final int baseValue;
	/** Weapons storage */
    @Getter
	private final IWeaponStorage weaponStorage;
    @Getter
    private int propertyTax = 5; // todo: andi 4/22/15: initialize through config properties
    @Getter
	private IStorage storage;
	@ObjectPropertyType(ISteward.class)
	private ObjectProperty<ISteward> steward;
    @Getter
	private IBalanceSheet balanceLastWeek;
    @Getter
	private IBalanceSheet currentWeek;
    @Getter
	@Autowired
	private IAutomatedTrading officeTrading;
	
	private IntegerProperty capacity = new SimpleIntegerProperty(this, "capacity", MAX_CAPACITY);
    @XStreamOmitField
	private BooleanBinding storageManagerPresent = null;
    // TODO aho Dec 7, 2013: consider capacity of warehouses

	public TradingOffice(IPlayer player, ICity city, int value){
		this.owner=player;
		this.city = city;
		baseValue = value;
        steward = new SimpleObjectProperty<>(this, "steward", null);
        weaponStorage = new WeaponStorage();
	}
	@PostConstruct
	private void init() {
		if( storage == null) {
			storage = buildingFactory.createStorage(owner, city);
		}
		if (!containsWare(EWare.GRAIN)) {
			initWares();
		} else {
			bindWares();
		}
		if (balanceLastWeek == null) {
			balanceLastWeek = interactionFactory.createInitialBalanceSheet(city, owner);
		}
		if (currentWeek == null) {
			currentWeek = interactionFactory.createBalanceSheetFromPreviousWeek((BalanceSheet) balanceLastWeek);
		}
		clientServerEventBus.register(this);
	}



	@PreDestroy
	public void destroy() {
		unbindAllAmounts();
		clientServerEventBus.unregister(this);
	}
	/**
	 * Init the amount of wares available in the city
	 * This method is protected so it can be overriden by subclasses for testing
	 */
	private void initWares() {
		for (EWare ware : EWare.values()) {
				addNewWare(ware, 0);
		}

	}
	private void bindWares() {
		for (EWare ware : EWare.values()) {
			bindStoredAmount(getWare(ware).amountProperty());
		}
	}


	@Override
	public IntegerProperty capacityProperty() {
		return capacity;
	}



	@Override
	public boolean hasWeapons() {
		return weaponStorage.hasWeapons();
	}
	/**
	/**
	 * {@inheritDoc}
	 * This method is thread save.
	 */
	@Override
	public int move(IWeapon weapon, int amount) {
		synchronized (weapon) { // There is still a problem with networked solution where the model of the city is replicated.
			return weaponStorage.update(weapon, amount);
		}
	}
	/**
	 * Compute the average price for ware that is to be purchased in the specified
	 * amount. The price is computed based on the amount that is available in the city.
	 * @param ware for which the average price is to be computed
	 * @param amount amount that is bought
	 * @return
	 */
	@Override
	protected int computeAVGPrice(IWare ware, int amount) {
		int available = city.getWare(ware).getAmount();
		return ware.computeSellPrice(available, amount);
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int move(IWare ware, int amount, int avgPrice) {
		synchronized (ware) { // There is still a problem with networked solution where the model of the city is replicated.
			AmountablePrice<IWare> amounted = getWare(ware);
			if (amount<0 && containsWare(ware) && amounted!=null && -amount>amounted.getAmount()){ // remove more than is available
				amount = -amounted.getAmount();
			}
			if (amount<0 && !containsWare(ware)){
				amount = 0;
			}
			if (!containsWare(ware)){ // If the ware does not exist add it with amount 0
				new IllegalStateException("All wares should be initialized, allowed in test").printStackTrace();
			}
			if (amount>0){ // amount was bought
				amounted.add(amount, avgPrice);
			} else { // amount sold
				amounted.remove(amount);
			}
			return amount;
		}
	}

	@Override
	public ISteward getSteward() {
		return steward.get();
	}
	@Override
	public void setSteward(ISteward steward) {
		this.steward.set(steward);
	}
    @Override
    public ObjectProperty<ISteward> stewardProperty() {
        return steward;
    }

//	@Override
//	public int getPropertyTax() {
//		// TODO check the values for this, make this dependent on the difficulty
//		return 5;
//	}
	@Subscribe
	public void handleWeeklyUpdates(PeriodicalTimeWeekUpdate event) {

		IBalanceSheet newWeek = interactionFactory.createBalanceSheetFromPreviousWeek((BalanceSheet) currentWeek);
        balanceLastWeek = currentWeek;
		currentWeek = newWeek;
	}

	@Override
	public int move(IWare ware, int amount) {
		return super.move(ware, amount, owner);
	}

	@Override
	public int getValue() {
		return baseValue;
	}


    @Override
    public BooleanBinding storageManagerPresentBinding() {
        if (storageManagerPresent == null) {
            storageManagerPresent = new BooleanBinding() {
                {
                    super.bind(steward);
                }

                @Override
                protected boolean computeValue() {
                    return steward.get() != null;
                }
            };
        }
        return storageManagerPresent;
    }


}
