package ch.sahits.game.openpatrician.model.city;

import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IShip;
import java.util.Optional;
import org.joda.time.DateTime;

import java.util.List;

/**
 * Interface defining the model of the ship yard.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Mar 9, 2013
 *
 */
public interface IShipyard {
    /**
     * Retrieve the city the shipyard belongs to.
     * @return
     */
    ICity getCity();
	/**
	 * Retrieve an immutable list of the ships that are ordered
	 * to be built on this shipyard.
	 * @return immutable list of ship due dates.
	 */
	List<IShipDueDate> getShipBuildingList();
	/**
	 * Retrieve the date when the building of a new ship will be finished.
	 * @param type of the ship to be built
	 * @return end date of the building
	 */
	DateTime getBuildCompleteDate(EShipType type);
//	/**
//	 * Order a ship of given type to be build. The costs are not deduced.
//	 * This method is called when the ship building is actually ordered.
//	 * @param type of the ship to build
//	 * @param owner of the ship to be built
//	 * @return date when the ship will be finished.
//	 */
//	DateTime orderShipBuilding(EShipType type, IPlayer owner);

    /**
     * Add an order for building a new ship to the list.
     * @param dueDate when the ship is finished building
     * @param task that is executed on completion.
     */
    void addShipBuildingOrder(IShipDueDate dueDate, Optional<TimedTask> task);

    /**
     * Add an order for repairing a ship to the list.
     * @param dueDate when the ship is finished repairing
     * @param task that is executed on completion.
     */
    void addShipRepairOrder(IShipDueDate dueDate, Optional<TimedTask> task);
    /**
     * Add an order for refitting a ship to the list.
     * @param dueDate when the ship is refitted
     * @param task that is executed on completion.
     */
    void addShipRefitOrder(IShipDueDate dueDate, Optional<TimedTask> task);
	/**
	 * Update the state with a building order. This method is used for synchronizing
	 * multiple instances of the same state in distributed systems (multiplayer).
	 * A task added in this manner belongs to another player and cannot be canceled.
	 * @param dueDate ship due date
	 */
	void addShipBuildingOrder(IShipDueDate dueDate);
	/**
	 * Cancel the order of the building, finished on the given date.
	 * @param finish date
	 */
	void cancelShipBuildingOrder(DateTime finish);
	/**
	 * Retrieve the list of all ships that are to be repaired.
	 * @return immutable list of all the repairing ships
	 */
	List<IShipDueDate> getShipRepairList();
//	/**
//	 * Order the repair of the ship. The costs for the repair not are deduced.
//     * With each repair order the crew gets more expierianced.
//	 * @param ship to be repaired.
//	 */
//	void repair(IShip ship);
	/**
	 * Add a ship to the repair list. This method is used for synchronizing
	 * multiple instances of the same state in distributed systems (multiplayer).
	 * A task added in this manner belongs to another player and cannot be canceled.
	 * @param dueDate ship due date
	 */
	void addShipToRepairList(IShipDueDate dueDate);
	/**
	 * Cancel the repair of the ship. No damage will be repaired
	 * @param ship which's repair order is to be canceled
	 */
	void cancelRepair(IShip ship);
//	/**
//	 * Refit the ship to the new level. The costs for the upgrade are <strong>not</strong> deduced.
//	 * @param ship to be refitted
//	 * @param upgradeLevel level to which the ship should be upgraded.
//	 */
//	void refit(IShip ship, EShipUpgrade upgradeLevel);
//	/**
//	 * Add a ship to the refit list. This method is used for synchronizing
//	 * multiple instances of the same state in distributed systems (multiplayer).
//	 * A task added in this manner belongs to another player and cannot be canceled.
//	 * @param dueDate ship due date
//	 * @param dueDate
//	 */
	void addToRefitList(IShipDueDate dueDate);
	/**
	 * Calculate the repair costs.
	 * @param repairTime
	 * @param materialCosts
	 * @return
	 */
	int calculateRepairCosts(int repairTime, int materialCosts);
	/**
	 * Calculate the material costs for the repair.
	 * @param ship
	 * @param damage
	 * @return
	 */
	int calculateRepairMaterialCosts(IShip ship, int damage);
	/**
	 * Calculate the labor costs for the refit based on the ship type and how many levels
	 * should be upgraded.
	 * @param type
	 * @param levels
	 * @return
	 */
	int calculateRefitCosts(EShipType type, int levels);

    /**
     * Calculate the time in days it takes for refitting.
     * @param type destination level
     * @return time in days.
     */
    int calculateRefitTime(EShipType type);
	/**
	 * Retrieve a list of all the types of ships this yard can build.
	 * @return
	 */
	EShipType[] getBuildableShipTypes();
	/**
	 * Calculate the labor costs for building a new ship of the indicated type.
	 * @param type of the ship
	 * @return costs.
	 */
	int calculateConstructionCosts(EShipType type);
	/**
	 * Calculate the build time for the ship of type.
	 * @param type
	 * @return
	 */
	int calculateBuildTime(EShipType type);
	/**
	 * Calculate the repair time for the ship of type.
	 * @param type
	 * @return
	 */
	int calculateRepairTime(EShipType type);
}
