package ch.sahits.game.openpatrician.model.city;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.util.PropertyLoader;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * This class defines the amount of wares consumed by each populace class.
 * The amount is per week. The sizes are specified in the ware specific sizes
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 28, 2011
 *
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class PopulationConsume {
	/** Store the amounts */
	@MapType(key=EWare.class, value = Map.class)
	private Map<EWare, Map<EPopulationClass,Double>> needs = new HashMap<EWare, Map<EPopulationClass,Double>>();
	/**
	 * Initialize the needs structure based on a property file
	 * @param propFileName
	 */
	public PopulationConsume(String propFileName) { // different difficulty levels different property files
		Properties props = PropertyLoader.loadProperties(propFileName);
		EWare[] wares = EWare.values();
		EPopulationClass[] classes = EPopulationClass.values();
		for (EWare ware : wares) {
			final String wareName = ware.name();
			Map<EPopulationClass,Double> inner = new HashMap<EPopulationClass, Double>();
			double val;
			for (EPopulationClass popClass : classes) {
				String key = wareName+"_"+popClass.name();
				if (popClass == EPopulationClass.BEGGAR) {
					val = 0;
				} else {
					val = Double.parseDouble(props.getProperty(key));
				}
				inner.put(popClass, val);
			}
			needs.put(ware, inner);
		}
	}
	
	public PopulationConsume(){
		this("populationConsume_default.properties");
	}
	/**
	 * Retrieve the amount needed of a ware by a population class of the size
	 * @param ware to be requested
	 * @param popClass population class
	 * @param popSize number of people of the class
	 * @return amount needed per week
	 */
	public double getNeed(IWare ware,EPopulationClass popClass, int popSize){
		double d = needs.get(ware).get(popClass); // This is the amount for 1000
		d *= popSize/1000.0;
		return d;
	}

    /**
     * Calculate the weekly overall consumption of a ware in a city.
     * @param ware that's consumption should be retrieved
     * @param city in which the consumption is checked.
     * @return amount of the ware consumed within a week.
     */
    public double getWeeklyConsumption(IWare ware, ICity city) {
        double amount = 0;
        for (EPopulationClass popClass : EPopulationClass.values()) {
            amount += getNeed(ware, popClass, city.getPopulation(popClass));
        }
        return amount;

    }
	
}
