package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.ILoanRequest;
import ch.sahits.game.openpatrician.model.city.ILoaner;
import ch.sahits.game.openpatrician.model.impl.Citizen;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 30, 2014
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class LoanerState implements ILoaner{
    private static final int MAX_WEEKS_FOR_REQUEST = 4;
    private static final int MAX_REQUESTS = 6;
    private static final int MAX_LOAN_PROPOSAL = 3;
    @Getter
    private final ICity city;
    @Getter
    @ListType(IDebt.class)
    private List<IDebt> debts = new ArrayList<>();
    @Getter
    @ListType(ILoanRequest.class)
    private List<ILoanRequest> requests = new ArrayList<>();
    @Getter
    @ListType(ILoanProposal.class)
    private List<ILoanProposal> loanProposals = new ArrayList<>();
    @Autowired
    private Date date;
    @Autowired
    @XStreamOmitField
    private Random rnd;

    private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
    private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");

    public LoanerState(ICity city) {
        this.city = city;
    }

    @Override
    public void update() {
       clearOldRequests();
        while (requests.size() < MAX_REQUESTS) {
            createNewLoanRequest();
        }
        while (loanProposals.size() < MAX_LOAN_PROPOSAL) {
            createNewLoanProposal();
        }
    }

    private void createNewLoanProposal() {
        int duration = rnd.nextInt(30) + 5;
        double interesst = rnd.nextDouble()+1; // [1,2]
        interesst = interesst/100 + 1; // [1.01, 1.02]
        int amount = rnd.nextInt(2500)*10 + 5000;
        LoanProposal proposal = LoanProposal.builder()
                .amount(amount)
                .durationInWeeks(duration)
                .interest(interesst)
                .build();
        loanProposals.add(proposal);
    }

    private void createNewLoanRequest() {
        DateTime now = date.getCurrentDate();
        int duration = rnd.nextInt(50) + 10;
        int rankIndexBound = ESocialRank.values().length - 2; // no mayor or alderman
        final int rankIndex = rnd.nextInt(rankIndexBound);
        ESocialRank rank = ESocialRank.values()[rankIndex];
        Citizen citizen = new Citizen(lastNameLoader.getRandomName(), firstNameLoader.getRandomName(), rank, city);
        int amount = (rnd.nextInt(10000) + 1000) * (rankIndex + 1);
        LoanRequest request = LoanRequest.builder()
                .citizen(citizen)
                .durationInWeeks(duration)
                .issued(now)
                .sum(amount)
                .build();
        requests.add(request);
    }

    private void clearOldRequests() {
        DateTime now = date.getCurrentDate();
        DateTime before4Weeks = now.minusWeeks(MAX_WEEKS_FOR_REQUEST);
        for (Iterator<ILoanRequest> iterator = requests.iterator(); iterator.hasNext(); ) {
            ILoanRequest request = iterator.next();
            if (request.issued().isBefore(before4Weeks)) {
                iterator.remove();
            }
        }
    }


    @Override
    public void addDebt(IDebt debt) {
        DateTime now = date.getCurrentDate();
        Preconditions.checkArgument(debt.getDueDate().isAfter(now));
        debts.add(debt);
    }
}
