package ch.sahits.game.openpatrician.model.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.GameFactory;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.util.ClassChecker;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import static com.google.common.collect.Lists.newArrayList;
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class Player implements IPlayer {
	@ListType(IShip.class)
	private final ObservableList<IShip> fleet;
	private final long UUID;
    @Getter
	private ESocialRank rank; // TODO this has to be updated monthly
    @Getter
	private ICareer careerLevel; // TODO this has to be updated
	private IPersonalData personalData;
	private ISpouseData spouse;
    @Getter
	private final ICompany company;
	@Getter
	@Setter
	private MapSegmentedImage segmentedMap;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;



	public Player(ICity homeTown, IPersonalData personalData, long cash,ESocialRank rank, GameFactory gameFactory) {
		super();
		fleet = FXCollections.observableArrayList();
		this.rank=rank;
		careerLevel=EEconomicCareer.INEXPERIENCED;
		this.personalData=personalData;
		company = gameFactory.createCompany(this, homeTown, cash);
		UUID = new Random().nextLong();
	}
	@PostConstruct
	private void init() {
	}

	@Override
	public String getName() {
		return personalData.getName();
	}

	@Override
	public String getLastName() {
		return personalData.getLastName();
	}

	@Override
	public ICity getHometown() {
		return company.getHomeTown();
	}

	@Override
	public List<IShip> getFleet() {
		return Collections.unmodifiableList(fleet);
	}

	@Override
	public boolean isMale() {
		return personalData.isMale();
	}

	@Override
	public void addShip(IShip ship) {
		fleet.add(ship);
	}

	@Override
	public void removeShip(IShip ship) {
		fleet.remove(ship);
	}

	@Override
	public long getCash() {
		return company.getCash();
	}
	/**
	 * Add or subtract some cash
	 * @param diff amount of money that is transferred
	 */
	@Override
	public void updateCash(int diff){
		company.updateCash(diff);
	}

	@Override
	public List<IBuilding> findBuildings(ICity city) {
		ArrayList<IBuilding> result = new ArrayList<IBuilding>();
		for (IBuilding building : city.getBuildings()) {
			if (building.getOwner()==this){
				result.add(building);
			}
		}
		return result;
	}

	@Override
	public ITradingOffice findTradingOffice(ICity city) {
		for (IBuilding building : city.getBuildings()) {
			if (building.getOwner()==this && building instanceof ITradingOffice){
				return (ITradingOffice) building;
			}
		}
		return null;
	}

	@Override
	public <T extends IBuilding> List<T> findBuildings(ICity city,Class<T> buildingClass) {
		List<T> result = new ArrayList<>();
		for (IBuilding buidlding : city.getBuildings()) {
			ClassChecker checker = new ClassChecker(buidlding.getClass());
			if (checker.extendsClass(buildingClass) || checker.implementsInterface(buildingClass)){
				result.add((T)buidlding);
			}
		}
		return result;
	}

	/**
	 * Marry a spouse
	 * @param spouse
	 */
	public void marry(ISpouseData spouse){
		this.spouse=spouse;
	}
	/**
	 * Widow and become unmarried again
	 */
	public void widow(){
		spouse=null;
	}

	@Override
	public IPersonalData getPersonalData() {
		return personalData;
	}

	@Override
	public ISpouseData getSpouseData() {
		return spouse;
	}

	@Override
	public void updateRank(ESocialRank rank) {
		this.rank=rank;
	}

	@Override
	public List<IShip> findShips(ICity city) {
		List<IShip> ships = newArrayList();
		for (IShip ship : fleet) {
			if (ship.getLocation().equals(city.getCoordinates())) {
				ships.add(ship);
			}
		}
		return ships;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		Player player = (Player) o;

		if (UUID != player.UUID) {
			return false;
		}
		if (personalData != null ? !personalData.equals(player.personalData) : player.personalData != null) {
			return false;
		}

		return true;
	}

	@Override
	public int hashCode() {
		int result = (int) (UUID ^ (UUID >>> 32));
		result = 31 * result + (personalData != null ? personalData.hashCode() : 0);
		return result;
	}

    @Override
    public void receiveSum(long amount) {
        getCompany().updateCash(amount);
    }
}
