package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import com.google.common.base.Preconditions;
import com.google.common.collect.Range;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.joda.time.DateTime;
import org.joda.time.Years;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.Random;

/**
 * Implementation of the captain.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 26, 2013
 *
 */
@Component
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class CaptainState implements ICaptain {
	@Autowired
	private Date date;
	@Autowired
	@XStreamOmitField
	private Random rnd;
	private DateTime birthDate;
	private String name;

	private int salary;
	private int tradingSkill;
	private int navigationSkill;
	private int fightingSkill;

	private final static Range<Integer> skillRange = Range.closed(0, 5);


	/*
	 * TODO use captain names
	 */
	private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
	private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");

	@PostConstruct
	private void init() {
		name = firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName();
		birthDate = date.getCurrentDate().minusYears(rnd.nextInt(15)+25)
				.minusMonths(rnd.nextInt(12))
				.minusDays(rnd.nextInt(30));
		setFightSkillLevel(rnd.nextInt(6));
		setTradingSkillLevel(rnd.nextInt(6));
		setNavigationSkillLevel(rnd.nextInt(6));
		//TODO: Improve this calculation by balancing it and by using the captain's age
		setSalary(rnd.nextInt(6)+10+tradingSkill*2+navigationSkill*2+fightingSkill*2);
	}


	@Override
	public int getAge() {
		return Years.yearsBetween(birthDate, date.getCurrentDate()).getYears();
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public int getSalary() {
		return salary;
	}

	@Override
	public int getTradingSkillLevel() {
		return tradingSkill;
	}

	@Override
	public int getNavigationSkillLevel() {
		return navigationSkill;
	}

	@Override
	public int getFightSkillLevel() {
		return fightingSkill;
	}

	/**
	 * Set the new trading skill level between [0..5]
	 * @param level
	 */
	public void setTradingSkillLevel(int level) {
		Preconditions.checkArgument(skillRange.contains(level), "Level must be within [0,5]");
		tradingSkill = level;
	}

	/**
	 * Set the new navigation skill level between [0..5]
	 * @param level
	 */
	public void setNavigationSkillLevel(int level) {
		Preconditions.checkArgument(skillRange.contains(level), "Level must be within [0,5]");
		navigationSkill = level;
	}

	/**
	 * Set the new fight skill level between [0..5]
	 * @param level
	 */
	public void setFightSkillLevel(int level) {
		Preconditions.checkArgument(skillRange.contains(level), "Level must be within [0,5]");
		fightingSkill = level;
	}

	/**
	 * Set the new salary
	 * @param salary
	 */
	public void setSalary(int salary) {
		this.salary = salary;
	}

}
