package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import com.google.common.base.Preconditions;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.ReadOnlyIntegerPropertyBase;

/**
 * Ecapsulate the price computation.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 25, 2013
 *
 */
@ClassCategory(EClassCategory.HANDLER)
public class ComputablePrice {
	private final IPriceCalculation calculation = new SimplePriceCalculation();
	private final ReadOnlyIntegerProperty maxValueBuy;
	private final ReadOnlyIntegerProperty minValueBuy;
	private final ReadOnlyIntegerProperty maxValueSell;
	private final ReadOnlyIntegerProperty minValueSell;
	private final ReadOnlyIntegerProperty saturation;
	private final String name;

	
	public ComputablePrice(final int maxValueBuy, final int minValueBuy, final int maxValueSell,
			final int minValueSell, final int saturation, String name) {
		super();
		this.maxValueBuy = new ReadOnlyIntegerPropertyBase(){

			@Override
			public Object getBean() {
				return ComputablePrice.this;
			}

			@Override
			public String getName() {
				return "maxValueBuy";
			}

			@Override
			public int get() {
				return maxValueBuy;
			}
		};
		this.minValueBuy = new ReadOnlyIntegerPropertyBase(){

			@Override
			public Object getBean() {
				return ComputablePrice.this;
			}

			@Override
			public String getName() {
				return "minValueBuy";
			}

			@Override
			public int get() {
				return minValueBuy;
			}
		};
		this.maxValueSell = new ReadOnlyIntegerPropertyBase(){

			@Override
			public Object getBean() {
				return ComputablePrice.this;
			}

			@Override
			public String getName() {
				return "maxValueSell";
			}

			@Override
			public int get() {
				return maxValueSell;
			}
		};
		this.minValueSell = new ReadOnlyIntegerPropertyBase(){

			@Override
			public Object getBean() {
				return ComputablePrice.this;
			}

			@Override
			public String getName() {
				return "minValueSell";
			}

			@Override
			public int get() {
				return minValueSell;
			}
		};
		this.saturation = new ReadOnlyIntegerPropertyBase(){

			@Override
			public Object getBean() {
				return ComputablePrice.this;
			}

			@Override
			public String getName() {
				return "saturation";
			}

			@Override
			public int get() {
				return saturation;
			}
		};
		this.name = name;
	}
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getMaxValueBuy()
	 */
	public int getMaxValueBuy() {
		return maxValueBuy.get();
	}
	public ReadOnlyIntegerProperty maxValueBuyProperty() {
		return maxValueBuy;
	}
	/**
	 * Retrieve the highest price of this ware when only on item is available
	 * @return
	 */
	public int getMaxValueSell() {
		return maxValueSell.get();
	}
	public ReadOnlyIntegerProperty maxValueSellProperty() {
		return maxValueSell;
	}
	/**
	 * Retrieve the lowest price this ware is soled (at market saturation)
	 * @return
	 */
	public int getMinValueSell() {
		return minValueSell.get();
	}
	public ReadOnlyIntegerProperty minValueSell() {
		return minValueSell;
	}

	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getMinValueBuy()
	 */
	public int getMinValueBuy() {
		return minValueBuy.get();
	}
	public ReadOnlyIntegerProperty minValueBuy() {
		return minValueBuy;
	}
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getMarketSaturation()
	 */
	public int getMarketSaturation() {
		return saturation.get();
	}
	public ReadOnlyIntegerProperty marketSaturationProperty() {
		return saturation;
	}
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#computeSellPrice(int, int)
	 */
	@Deprecated
	public int computeSellPrice(int availableAmount, int amountToSell) {
		return internalSellPriceCalculation(availableAmount, amountToSell);
	}
	/**
	 * Internal calculation of the sell price.
	 * @param availableAmount
	 * @param amountToSell
	 * @return
	 */
	int internalSellPriceCalculation(int availableAmount, int amountToSell) {
		Preconditions.checkArgument(amountToSell > 0, "The amount to sell must be larger than 0 (was "+availableAmount+") for "+name);
		Preconditions.checkArgument(availableAmount >= 0, "The available amount must be positive for "+name);
		if (availableAmount>=getMarketSaturation()){
			return getMinValueSell();
		}
		int min = getMinValueSell();
		int max = getMaxValueSell();
		// Take amounts
		int firstItemPrice = (int)Math.rint(calculation.computePrice(min, max, availableAmount,getMarketSaturation(),0,null,null));
		if (amountToSell>1){
			int lastItemPrice =  (int)Math.rint(calculation.computePrice(min, max, availableAmount+amountToSell,getMarketSaturation(),0,null,null));
			return (int) Math.rint((firstItemPrice+lastItemPrice)/2.0);
		} else {
			return firstItemPrice;
		}
	}

	@Deprecated
	public int computeBuyPrice(int availableAmount, int amountToBuy) {
		return internalBuyPriceCalculation(availableAmount, amountToBuy);
	}
	/**
	 * Internal method to calculate the buy price.
	 * @param availableAmount
	 * @param amountToBuy
	 * @return
	 */
	int internalBuyPriceCalculation(int availableAmount, int amountToBuy) {
		Preconditions.checkArgument(amountToBuy > 0, "The amount to buy must be larger than 0 for "+name);
		Preconditions.checkArgument(availableAmount >= 0, "The available amount must be positive for "+name);
		if (availableAmount==0){
			return 0;
		}
		if (availableAmount>=getMarketSaturation()){
			return getMinValueBuy();
		}
		int min = getMinValueBuy();
		int max = getMaxValueBuy();
		// Take amounts
		int firstItemPrice =  (int)Math.rint(calculation.computePrice(min, max, availableAmount,getMarketSaturation(),0,null,null));
		if (amountToBuy>1){
			// TODO make sure the value is not above the limit
			int lastItemPrice =  (int)Math.rint(calculation.computePrice(min, max, availableAmount-amountToBuy,getMarketSaturation(),0,null,null));
			return (int) Math.rint((firstItemPrice+lastItemPrice)/2.0);
		} else {
			return firstItemPrice;
		}
	}
	/**
	 * Integer binding for the buy price.
	 * @param availableAmount
	 * @param amountToBuy
	 * @return
	 */
	public int buyPrice(final IntegerProperty availableAmount, final IntegerBinding amountToBuy) {
//		IntegerBinding binding = new IntegerBinding() {
//			{
//				super.bind(availableAmount, amountToBuy);
//			}
//			
//			@Override
//			protected int computeValue() {
				return internalBuyPriceCalculation(availableAmount.get(), amountToBuy.get());
//			}
//		};
//		return binding;
	}
	
	public int sellPrice(final IntegerProperty availableAmount, final IntegerBinding amountToSell) {
//		IntegerBinding binding = new IntegerBinding() {
//			{
//				super.bind(availableAmount, amountToSell);
//			}
//			
//			@Override
//			protected int computeValue() {
				return internalSellPriceCalculation(availableAmount.get(), amountToSell.get());
//			}
//		};
//		return binding;
	}
}
