package ch.sahits.game.openpatrician.model;

import ch.sahits.game.event.data.ClockTickIntervalChange;
import ch.sahits.game.openpatrician.IInvalidatable;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import java.util.Calendar;

/**
 * Representation of the date within the game.
 * This class is thread safe.
 * The Date is implemented as a singleton. However there
 * is no getInstance method to retrieve the once created instance,
 * so it must be referenced elsewhere.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
@Component
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN,EClassCategory.SINGLETON_BEAN})
public class Date implements IInvalidatable {
	@XStreamOmitField
	private final Logger logger = LogManager.getLogger(getClass());

	public static final int DEFAULT_INVALID_START_YEAR = 0;
	/** Tick update in minutes */
	@Getter
    private int tickUpdate;
	/** Current date */
	@ObjectPropertyType(DateTime.class)
	private ObjectProperty<DateTime> cal = new SimpleObjectProperty<>(this, "cal", null);;
	@Getter
    private DateTime startDate;
	/** Start year */
    @Getter
	private Integer startYear = null;
	/** Lock for guaranteeing thread safety */
	private static Object lock = new Object();
	@XStreamOmitField
	private StringBinding dayDateBinding = null;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;
	@Autowired
	@XStreamOmitField
	private Locale locale;
	@Autowired
	@XStreamOmitField
	private MessageSource messageSource;
	/**
	 * Initialize the date with start year.
	 * The initial date is 13th July of the start year.
	 * The default tick update is 5 minutes.
	 */
	public Date(int year){
		this();
		setStartYear(year);
	}


	public Date() {
		tickUpdate = 5;
	}
//    @PostConstruct
//    private void init() {
//        cal.addListener((observable, oldValue, newValue) -> {
//            if (dayDateBinding != null) {
//                dayDateBinding.invalidate();
//            }
//        });
//    }
	public void resetStartYear(int startYear) {
		System.out.println("+++++++++++++++++++ set start year: "+startYear);
		this.startYear=startYear;
		updateTime(new DateTime(startYear, 7, 13, 0, 0));
		startDate = getCurrentDate();
	}
	public void setStartYear(int startYear) {
		Preconditions.checkState(checkStartYear(startYear), "Date may only be initialized once");
		resetStartYear(startYear);
	}
	private boolean checkStartYear(int startYear) {
		if (this.startYear == null) {
			return true;
		}
		if (this.startYear == DEFAULT_INVALID_START_YEAR) {
			return true;
		}
		return false;
	}
	/**
	 * Constructor for testing purposes only
	 * @param cal
	 */
	Date(DateTime cal){
		updateTime(cal);
		startYear = cal.getYear();
		tickUpdate = 5;
	}

    /**
     * Change the time.
     * @param newTime
     */
	public void updateTime(DateTime newTime) {
		if (cal.get() != null && !isToday(newTime) && dayDateBinding != null) {
			dayDateBinding.invalidate();
		}
		Preconditions.checkNotNull(newTime);
		cal.set(newTime);
    }

	/**
	 * Set the tick update in number of minutes
	 * @param minutes tick update
	 */
	public void setTickUpdate(int minutes){
		tickUpdate =minutes;
		final ClockTickIntervalChange event = new ClockTickIntervalChange();
		event.setInterval(tickUpdate);
		clientServerEventBus.post(event);
	}

	/**
	 * Retrieve the date of the start of the week in the form {Day of month}. {Month}.
	 * The week starts with Monday
	 * @return
	 */
	public String getStartOfWeek(){
		DateTime lastMon = getStartOfWeekInternal();
		return toShortDate(lastMon);
	}
	/**
	 * Retrieve the date of the start of the last week
	 * @return
	 */
	DateTime getStartOfWeekInternal() {
		// last week
		DateTime lastMonday = cal.get().minusWeeks(1);
		// first day
		lastMonday = lastMonday.minusDays(lastMonday.getDayOfWeek()-1);
		return lastMonday;
	}
	/**
	 * Convert the date into the form {Day in month}. {Month}
	 * @param c
	 * @return
	 */
	public String toShortDate(DateTime c) {
		final int day;
		final int month;
		synchronized (lock) {
			day = c.getDayOfMonth();
			month = c.getMonthOfYear();
		}
		StringBuilder sb = new StringBuilder();
		sb.append(day).append(". ");
		sb.append(messageSource.getMessage("month_short_"+month, new Object[]{}, locale.getCurrentLocal()));

		return sb.toString();
	}
	/**
	 * Retrieve the date of the end of the week in the form {Day of month}. {Month}.
	 * The week ends with Sunday
	 * @return
	 */
	public String getEndOfWeek(){
		DateTime lastSun = getEndOfWeekInternal();
		return toShortDate(lastSun);
	}
	/**
	 * Retrieve the date of the of the Sunday of last week
	 * @return
	 */
	DateTime getEndOfWeekInternal() {
		int cdow = cal.get().getDayOfWeek();
		DateTime lastDayOfWeek = cal.get().plusDays(7 - cdow);
		return lastDayOfWeek;
	}

	/**
	 * Create a string representation for the user interface of the
	 * form {Day of month}. {Month} {Year}.
	 * @return
	 */
	public String toDisplayString(){
		final int day;
		final int month;
		final int year;
		synchronized (lock) {
			day = cal.get().getDayOfMonth();
			month = cal.get().getMonthOfYear();
			year = cal.get().getYear();
		}
		return todisplayString(day, month, year);
	}

    /**
     * Bind the representation of the day as a String.
     * @return
     */
    public StringBinding dayDateBinding() {
          if (dayDateBinding == null) {
              dayDateBinding = new StringBinding() {
                  @Override
                  protected String computeValue() {
                      return toDisplayString();
                  }
              };
          }
        return dayDateBinding;
    }
	// TODO aho Jan 28, 2013: add utility with DateTime and formatter
	public String todisplayString(final int day, final int month, final int year) {
		StringBuilder sb = new StringBuilder();
		sb.append(day).append(". ");
		sb.append(messageSource.getMessage("month_short_" + month, new Object[]{}, locale.getCurrentLocal())).append(" ");
		sb.append(year);
		return sb.toString();
	}

    /**
     * Check if two dates represent the same day
     * @param date1
     * @param date2
     * @return
     */
    public boolean isSameDay(DateTime date1, DateTime date2) {   // todo: andi 4/6/15: refactor this into the date utility class
        return date1.toDateMidnight().equals(date2.toDateMidnight());
    }

    /**
     * Check if the date is today.
     * @param date
     * @return
     */
    public boolean isToday(DateTime date) {   // todo: andi 4/6/15: refactor this into the date utility class
        return date.toDateMidnight().equals(getCurrentDate().toDateMidnight());
    }
	/**
	 * Retrieve the immutable DateTime
	 * @return
	 */
	public DateTime getCurrentDate(){
		return cal.get();
	}
	/**
	 * Retrieve the immutable DateTime
	 * @return
	 */
	public ObjectProperty<DateTime> getCurrentDateProperty(){
		return cal;
	}
	/**
	 * Retrieve an index for the current weekday. Mondy is 0.
	 * @return
	 */
	public final int getWeekdayIndex(){
		int dayOfWeek = cal.get().getDayOfWeek()+1;
		return (dayOfWeek-Calendar.MONDAY)%7;
	}

    @Override
    public void invalidate() {
        dayDateBinding.invalidate();
        dayDateBinding = null;
        dayDateBinding();
    }
}
