package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.model.city.CityProduction;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.factory.StateFactory;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import com.google.common.annotations.VisibleForTesting;
import java.util.Optional;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import lombok.Setter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.TreeMap;

/**
 * This class encapsulates state information used by CityEngine about one city.
 * This class i package private since it should not be used by any other class than CityEngine.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 29, 2011
 *
 */
@Component
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class CityState {
	@XStreamOmitField
    private final Logger logger = LogManager.getLogger(getClass());

	@Autowired
	@XStreamOmitField
	private Random rnd;
	@Value("${ware.surplus.treshold}")
	private double surplusThreshold;

	/** Reference to the city model that is driven by this engine */
    @Getter
	private ICity city;
	/** Map of consumed wares, since only whole units can be consumed we need to store the fractions */
	@MapType(key=IWare.class, value=Double.class)
	private Map<IWare, Double> consumed = new HashMap<>();
	/** Map of produced wares, since only whole units can be consumed we need to store the fractions */
	@MapType(key=IWare.class, value=Double.class)
	private Map<IWare, Double> produced = new HashMap<>();
	/** Mock up to simulate the city production */
	@Autowired
	private CityProduction production;
	/** Tavern state for this city */
    @Getter
	private TavernState tavernState = null;
	/** Ship yard state fos the city */
    @Getter
	private IShipyard shipyardState = null;
	/** Provider for the consume numbers */
	@Autowired
	private PopulationConsume consume;
	@Autowired
	@XStreamOmitField
	private StateFactory stateFactory;
    @Getter
    @Setter
    @Autowired
    private CityWall cityWall;

	public CityState(ICity city) {
		this.city = city;
	}
	@PostConstruct
	private void initCity() {
		if (shipyardState == null) {
			shipyardState = stateFactory.createShipYard(city);
		}
		EWare[] wares = EWare.values();
		for (EWare ware : wares) {
			consumed.put(ware, 0.0);
			produced.put(ware, 0.0);
		}
		if (tavernState == null) {
			tavernState = stateFactory.createTavernState(city);
		}
		if (city.getCityState() == null) {
			city.setCityState(this);
		}
	}
	/**
	 * compute the wares consumed in the last 12 hours and remove them from the city
	 * This method is package private for testing purposes
	 */
	public void consumeWares() {
		EWare[] wares = EWare.values();
		EPopulationClass[] popClasses = EPopulationClass.values();
		for (EWare ware : wares) {
			double amount = consumed.get(ware);
			for (EPopulationClass popClass : popClasses) {
				amount += consume.getNeed(ware, popClass, city.getPopulation(popClass));
			}
			consumed.put(ware, amount);// does this produce ConcurrentModExc?
		}
		for (Entry<IWare, Double> entry : consumed.entrySet()) {
			if (entry.getValue()>=1){
				int whole = (int) Math.rint(Math.floor(entry.getValue()));
				consumed.put(entry.getKey(), entry.getValue()-whole); // does this produce ConcurrentModExc?
				city.move(entry.getKey(), -whole,null); // the city consumes the ware
			}
		}
	}

	/**
	 * Compute the wares produced in the last 12 hours and add the whole quatities to the city.
	 * This method is package private for testing purposes
	 */
	public void produceWares() {
		IWare[] wares = city.getEffectiveProduction();
		for (IWare ware : wares) {
			double amount = produced.get(ware);
			amount += production.getEfficientProduction(ware)/(7.0*2); // the amount is for a whole week
			produced.put(ware, amount);// does this produce ConcurrentModExc?
		}
		wares = city.getIneffectiveProduction();
		for (IWare ware : wares) {
			double amount = produced.get(ware);
			amount += production.getInefficientProduction(ware)/(7.0*2); // the amount is for a whole week
			produced.put(ware, amount);// does this produce ConcurrentModExc?
		}
		for (Entry<IWare, Double> entry : produced.entrySet()) {
			if (entry.getValue()>=1){
				int whole = (int) Math.rint(Math.floor(entry.getValue()));
				produced.put(entry.getKey(), entry.getValue()-whole); // does this produce ConcurrentModExc?
				city.move(entry.getKey(), whole,null); // The city produces the ware
			}
		}
	}
	/**
	 * Find the ware which is consumed the most and which is missing for the longest time.
	 * @return
	 */
	public Optional<IWare> findMostNeededWare() {
		Map<IWare, DateTime> missing = city.getMissingWares();
		DateTime earliest = null;
		IWare result = null;
		for (Entry<IWare, DateTime> entry : missing.entrySet()) {
			if (earliest == null ||
					entry.getValue().isBefore(earliest)) {
				earliest = entry.getValue();
				result = entry.getKey();
			}
		}
        if (result == null) { // All wares are availalbe => Look which one will run out first
            Map<IWare, Integer> available = getSortedWareAvailabilityMap();
            IWare wareRunninOutFirst = null;
            int runsOutInDays = Integer.MAX_VALUE;
            for (Entry<IWare, Integer> entry : available.entrySet()) {
                IWare ware = entry.getKey();
                double consumptionPerWeek = consume.getWeeklyConsumption(ware, city);
                double productionPerWeek = production.getProduction(ware);
                double dailyConsumtion = (productionPerWeek - consumptionPerWeek)/7;
                if (dailyConsumtion >= 0) {
//                    System.out.println("DailyConsumtion of " + ware.name() + " is " + dailyConsumtion);
                    continue; // more ware is produced than consumed
                }
                int days = -(int)(entry.getValue()/dailyConsumtion);
//                System.out.println("DailyConsumption: " + dailyConsumtion + ", runsOutIn " + days);
                if (days < runsOutInDays) {
                    runsOutInDays = days;
                    wareRunninOutFirst = ware;
                }
            }
            result = wareRunninOutFirst;
        }

		return Optional.ofNullable(result);
	}
	/**
	 * Find the ware that the city sells near the minimal price.
	 * @return
	 */
	public Optional<IWare> findWareWithMostSurplus() {
		ArrayList<IWare> surplus = new ArrayList<IWare>();
		for (IWare ware : EWare.values()) {
			AmountablePrice<IWare> amount = city.getWare(ware);
			double actualPrice = amount.getAVGPrice();
			int minPrice = ware.getMinValueSell();
			double val = minPrice/actualPrice - 1;
			if (val<=surplusThreshold) {
				surplus.add(ware);
			}
		}
		if (surplus.isEmpty()) {
			return Optional.empty();
		} else {
			return Optional.of(surplus.get(rnd.nextInt(surplus.size())));
		}
	}

    /**
     * Find the wares and order them in the amount of their availability. If there
     * is a ware with the same availability, only one is added to the map.
     * @return
     */
    @VisibleForTesting
    Map<IWare, Integer> getSortedWareAvailabilityMap() {
        Comparator<IWare> comparator = new Comparator<IWare>() {
            @Override
            public int compare(IWare ware1, IWare ware2) {
                return city.getWare(ware1).getAmount() - city.getWare(ware2).getAmount();
            }
        };
        TreeMap<IWare, Integer> map = new TreeMap<>(comparator);
        for (IWare ware : EWare.values()) {
            map.put(ware, city.getWare(ware).getAmount());
        }
        map.size(); // this seems to be required to ensure the proper size is retrieved.
        return map;
    }
}
