package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.event.PersonLeavesTavernEvent;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.people.ITavernPerson;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.SimpleBooleanProperty;
import lombok.Getter;
import lombok.Setter;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import java.util.Random;

/**
 * Base class for people frequenting the tavern.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 20, 2013
 *
 */
@ClassCategory({EClassCategory.DEPENDS_ON_SERIALIZED_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class BaseTavernPerson implements ITavernPerson {
	/** Arrival date of the person in the tavern.
	 *  Is null as long as <code>isPresent</code> is false
	 *  for the first time
	 */
    @Getter
	private DateTime arrivalDate;
	@Value("${tavern.person.presence.max}")
    @Getter
	private int maxDaysPresent;
	@Value("${tavern.person.absence.max}")
    @Getter
	private int maxDaysAbsent;
	/** Indicating if the person is currently present */
	private BooleanProperty isPresent;
	@Autowired
	private Date date;
    @Autowired
    @Qualifier("serverClientEventBus")
	@XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Setter
	private ICity city;
	@Autowired
	private Random rnd;

	public BaseTavernPerson() {
        isPresent = new SimpleBooleanProperty(this, "isPresent", false);
	}
    @PostConstruct
    private void initArival() {
        arrivalDate = date.getStartDate().minusDays(rnd.nextInt(3));
    }


	@Override
	public boolean isPresent() {
		return isPresent.get();
	}
    @Override
    public BooleanProperty isPresentProperty() {
        return isPresent;
    }
	/**
	 * Person arrives in the tavern.
	 */
	@Override
	public final void arrive() {
		arrivalDate = date.getCurrentDate();
		isPresent.set(true);
	}
	/**
	 * Person leaves the tavern
	 */
	@Override
	public final void leave() {
		Preconditions.checkNotNull(city);
        isPresent.set(false);
        clientServerEventBus.post(new PersonLeavesTavernEvent(city, this));
	}
	@Override
	public int getNumberOfDaysSinceArrival() {
		DateTime now = date.getCurrentDate();
		Duration duration = new Duration(arrivalDate, now);
		return duration.toStandardDays().getDays();
	}
}
