package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.datastructure.GenericPair;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.OptionalGenericPairType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.AUpdatable;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.people.IInformant;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Optional;
import java.util.Random;

/**
 * Implementation of a informant. Every city can have a informant and is referenced
 * through it's TavernEngine.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 20, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@AUpdatable
public class InformantState extends BaseTavernPerson implements IInformant {

	@Autowired
	@XStreamOmitField
	private Random rnd;
	@Setter
	private int price;
    @Getter
	@Setter
	private String name;
    @Getter
	@OptionalGenericPairType(key=ICity.class, value = IWare.class)
	private Optional<GenericPair<ICity, IWare>> wareNeeded = Optional.empty();
    @Getter
	@OptionalGenericPairType(key=ICity.class, value = IWare.class)
    private Optional<GenericPair<ICity, IWare>> wareSurplus = Optional.empty();
    @Getter
	@OptionalGenericPairType(key=String.class, value = ICity.class)
    private Optional<GenericPair<String, ICity>> pirateLocation = Optional.empty();
	@Getter
	@Setter
	private boolean showMapIfAvailable;
	/*
	 * Use custom names
	 */
	private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
	private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");

	public InformantState() {
		name = firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName();
	}





	public int getPrice(IPlayer player) {
		if (showMapIfAvailable && player.getSegmentedMap() != null) {
			return price + rnd.nextInt(1000) + 1000;
		} else {
			return price;
		}

	}



	@Override
	public Optional<MapSegmentedImage> getMapSectionPart(IPlayer player) {
		if (player.getSegmentedMap() != null && showMapIfAvailable){
			MapSegmentedImage segmentedImage = player.getSegmentedMap();
			int index;
			do {
				index = rnd.nextInt(9);
			} while (segmentedImage.getSegmentIndices().contains(index));
			segmentedImage.addSegment(index);
			return Optional.of(segmentedImage);
		} else {
			return Optional.empty();
		}
	}

	@Override
	public void setPirateLocation(ISeaPirate pirate, ICity city) {
		if (pirate != null && city != null) {
			pirateLocation = Optional.of(new GenericPair<>(pirate.getName()+" "+pirate.getLastName(), city));
		} else {
			pirateLocation = Optional.empty();
		}
	}

	@Override
	public void setNeededWare(ICity city, IWare ware) {
		if (ware != null && city != null) {
			wareNeeded = Optional.of(new GenericPair<>(city, ware));
		} else {
			wareNeeded = Optional.empty();
		}
	}

	@Override
	public void setWareSurplus(ICity city, IWare ware) {
		if (ware != null && city != null) {
			wareSurplus = Optional.of(new GenericPair<>(city, ware));
		} else {
			wareSurplus = Optional.empty();
		}
	}
}
