package ch.sahits.game.openpatrician.model.ship.impl;

import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.collections.NonReplacableMap;
import ch.sahits.game.openpatrician.javafx.bindings.LateIntegerBinding;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.EShipSide;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IShipWeaponsLocation;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.geometry.Point2D;
import lombok.Getter;
import lombok.Setter;

import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;

import static com.google.common.collect.Maps.newHashMap;
@Prototype
public abstract class Ship implements IShip {
	/** Amount of barrels space a sailor needs */
	private final static double SAILOR_NEEDS_SPACE = 0.5;

	/** State of the design standard level */
    @Getter
	protected EShipUpgrade shipUpgradeLevel;
	/** Name of the ship */
    @Getter
    @Setter
	protected String name;
	/** Store the wares loaded on the ship together with their amount. The amount is ware specific*/
	@MapType(key=IWare.class, value = AmountablePrice.class)
	private Map<IWare,AmountablePrice<IWare>> loadedWare = new NonReplacableMap<>();
	/**
	 * Binding representing the current load.
	 */
    @XStreamOmitField
	private LateIntegerBinding loadBinding = createLoadBinding();

	private final double topSpeed;



    @XStreamOmitField
	private BooleanBinding passengerPresent = new BooleanBinding() {
		@Override
		protected boolean computeValue() {
			return passenger.isPresent();
		}
	};



	@MapType(key = IWeapon.class, value = Integer.class)
	private Map<IWeapon,Integer> loadedWeapons = newHashMap();
    @Getter
    @Setter
	private IShipWeaponsLocation shipWeaponsLocation;
    /** Initial value of the ship */
    private final int initialValue;
    //	/** List of the weapons on board together with their location */
//	private final ArrayList<WeaponsLocation> weapons = new ArrayList<WeaponsLocation>();
	@Getter
    private final int minNumberOfSailors;
    private final int maxSailors;
    @Getter
	private Optional<ICaptain> captian = Optional.empty();
    @Getter
    private Optional<IPerson> passenger = Optional.empty();
    /**
	 * Fitness of the ship. This is reduced as it is damaged
	 */
	protected int fitness = 100;

	private IntegerProperty nbSailors = new SimpleIntegerProperty(0);
    @Setter
    @Getter
	private Point2D location = null;
    @Setter
	private IShipOwner owner = null;
    @Getter
	private boolean piratingShip = false;
	@Getter
	@Setter
	private boolean available = true;
	/** Space occupied by wares not owned by the owner */
	@Getter
	@Setter
	private int occupiedSpace;
	/** Size in barrels */
	@Getter
	private int size; // todo: andi 12/6/14: make this final
	private int reductionLevel1;
	private int reductionLevel2;
	@Getter
	private final String uuid;

	protected Ship(Properties shipConfiguration) {
		this.initialValue = Integer.parseInt(shipConfiguration.getProperty("initialValue"));
		this.minNumberOfSailors = Integer.parseInt(shipConfiguration.getProperty("minNumberOfSailors"));
		this.maxSailors = Integer.parseInt(shipConfiguration.getProperty("maxSailors"));
		this.size = Integer.parseInt(shipConfiguration.getProperty("size"));
		this.reductionLevel1 = Integer.parseInt(shipConfiguration.getProperty("reductionLevel1"));
		this.reductionLevel2 = Integer.parseInt(shipConfiguration.getProperty("reductionLevel2"));
		this.topSpeed = Double.parseDouble(shipConfiguration.getProperty("speedkmperh"));
		uuid = UUID.randomUUID().toString();
	}

	/**
	 * Constructor in place for backward compatability.
	 * @param initialValue
	 * @param minNumberOfSailors
	 * @param maxSailors
	 */
	Ship(int initialValue, int minNumberOfSailors, int maxSailors) {
		this.initialValue = initialValue;
		this.minNumberOfSailors = minNumberOfSailors;
		this.maxSailors = maxSailors;
        this.topSpeed = 5;
//		size = 300;
		uuid = UUID.randomUUID().toString();
	}

    /**
     * Default constructor for special ship constructs like convoys.
     */
    protected Ship() {
        this.initialValue = 0;
        this.minNumberOfSailors = 0;
        this.maxSailors = 0;
        this.topSpeed = 5;
		uuid = UUID.randomUUID().toString();
    }


	@Override
	public int getMaxNumberOfSailors() {
		int availableSpace = getCapacity();
		int spaceForSailors = (int)Math.rint(availableSpace/SAILOR_NEEDS_SPACE);
		return Math.min(maxSailors, spaceForSailors);
	}


	@Override
	public void setCaptain(ICaptain captain) {
		this.captian = Optional.ofNullable(captain);
	}

	@Override
	public boolean isUpgradable() {
		return shipUpgradeLevel != EShipUpgrade.LEVEL2;
	}

	@Override
	public void upgrade() {
		EShipUpgrade[] levels = EShipUpgrade.values();
		for (int i = 0; i < levels.length; i++) {
			if (levels[i]==shipUpgradeLevel){
                shipUpgradeLevel = levels[i+1];
				break; // only one update allowed
			}
		}
	}


	@Override
	public Set<IWare> getLoadedWares() {
		return loadedWare.keySet();
	}

	/**
	 * {@inheritDoc}
	 * This method is not thread safe as it is only intended to be accessed by one thread at a time
	 */
	@Override
	public int load(IWare ware, int amount, int avgPrice) {
		final short sizeInBarrels = ware.getSizeAsBarrels();
		amount = Math.abs(amount);
		int cap = ensureCapacity(amount*sizeInBarrels);
		// Make sure that for wares in other sizes than barrels we only load
		// complete loads.
		if (cap!=amount && sizeInBarrels!=1){
			cap = (cap/sizeInBarrels)*sizeInBarrels;
		}
		AmountablePrice<IWare> available = getWare(ware);
		available.add(cap/ware.getSizeAsBarrels(), avgPrice);
		return cap/sizeInBarrels;
	}

	/**
	 * Check if the amount can be loaded
	 * @param amount
	 * @return
	 */
	private int ensureCapacity(int amount) {
		return Math.min(amount, getCapacity());
	}

	/**
	 * {@inheritDoc}
	 * This method is not thread safe as it is only intended to be accessed by one thread at a time
	 */
	@Override
	public int unload(IWare ware, int amount) {
		if (!loadedWare.containsKey(ware)){
			return 0; // nothing to unload
		}
		amount = Math.abs(amount);
		// convert to barrels
		amount = amount*ware.getSizeAsBarrels();
		final int loaded = loadedWare.get(ware).getAmount()*ware.getSizeAsBarrels();
		int unloaded = Math.min(loaded, amount);
		if (unloaded==loaded){ // unloaded completely
			AmountablePrice<IWare> available = loadedWare.get(ware);
			available.reset();
		} else {
			AmountablePrice<IWare> available = loadedWare.get(ware);
			available.remove(unloaded/ware.getSizeAsBarrels());
		}
		return unloaded/ware.getSizeAsBarrels();
	}


	private int internalLoadCalculation() {
		int sum=0;
		for (Entry<IWare,AmountablePrice<IWare>> entry : loadedWare.entrySet()) {
			int amount = entry.getValue().getAmount();
			int barrelSize = entry.getKey().getSizeAsBarrels();
			sum += amount*barrelSize;
		}
		return sum;
	}
	@Override
	public IntegerBinding getLoadBinding() {
		return loadBinding;
	}

	/**
	 * Clear all loaded wares. This method is only intended for testing
	 */
	protected void clearLoadedWares() {
		for (AmountablePrice<IWare> amountable : loadedWare.values()) {
			amountable.reset();
		}
	}

	@Override
	public AmountablePrice<IWare> getWare(IWare ware) {
		if (!loadedWare.containsKey(ware)){
			AmountablePrice<IWare> amountable = new AmountablePrice<IWare>();
			loadedWare.put(ware, amountable);
			loadBinding.bind(amountable.amountProperty());
		}
		return loadedWare.get(ware);
	}
	/**
	 * Retrieve the side of the next free slot where the weapon can be placed.
	 * check first port and then starboard. There is no free solt on either side,
	 * null will be returned.
	 * @param weapon to be placed
	 * @return {@link EShipSide#PORT}, {@link EShipSide#STARBOARD} or null
	 */
	@SuppressWarnings("unused")
	private EShipSide getNextFreeSide(IWeapon weapon){
		// TODO implement
		return null;
	}
	/**
	 * Retrieve the next free slot for the weapon on the side. The slots are checked
	 * from stern to bow. If there is no free slot a negative number will be returned
	 * @param weapon to be placed
	 * @param side to be checked {@link EShipSide#PORT} or {@link EShipSide#STARBOARD}
	 * @return slot index or negative number
	 */
	@SuppressWarnings("unused")
	private int getNextFreeSlot(IWeapon weapon, EShipSide side){
		// TODO implement
		return -1;
	}

	@Override
	public boolean hasWeapons() {
		return false; // TODO aho Jun 1, 2013: solve this through the ShipWeaponsLocation
	}
	@Override
	public int getValue() {
		return (int)Math.rint(initialValue*fitness/100.0);
	}


	@Override
	public int getNumberOfSailors() {
		return nbSailors.get();
	}


	@Override
	public void setNumberOfSailors(int nbSailors) {
		this.nbSailors.set(nbSailors);
	}

    @Override
    public IntegerProperty numberOfSailorsProperty() {
        return nbSailors;
    }

    @Override
	public void move(IWeapon weaponType, int amount){
		if (!loadedWeapons.containsKey(weaponType)) {
			loadedWeapons.put(weaponType, 0);
		}
		if (amount<0) {
			// remove
			boolean check = -amount<=loadedWeapons.get(weaponType);
			Preconditions.checkArgument(check, "Cannot remove more than available");
		}
		int amountable = loadedWeapons.get(weaponType);
		loadedWeapons.put(weaponType, amountable+amount);
	}

	@Override
	public int getWeaponAmount(IWeapon weaponType) {
		if (loadedWeapons.containsKey(weaponType)) {
			return loadedWeapons.get(weaponType);
		} else {
			return 0;
		}
	}
	/**
	 * Move a passenger to the ship. The ship may only contain one passanger.
	 * @param passanger
	 */
	@Override
	public void addPassenger(IPerson passanger) {
		Preconditions.checkArgument(!passenger.isPresent(), "Passenger already present");
		passenger = Optional.of(passanger);
		passengerPresent.invalidate();
	}
	@Override
	public IShipOwner getOwner() {
			return owner;
	}


	/**
	 * Passenger leave ship.
	 */
	@Override
	public void leavePassanger(){
		passenger = Optional.empty();
		passengerPresent.invalidate();
	}

	@Override
	public int getCapacity() {
		int passengerSpace = passenger.isPresent() ? 30 : 0;
		return size-getUpgradeSpaceReduction()-internalLoadCalculation()-getOccupiedSpace() - passengerSpace;
	}

	@Override
	public int getUpgradeSpaceReduction(){
		switch (shipUpgradeLevel) {
			case NONE:
				return 0;
			case LEVEL1:
				return reductionLevel1;
			case LEVEL2:
				return reductionLevel2;
			default:
				return 0;
		}
	}

	private LateIntegerBinding createLoadBinding() {
		return new LateIntegerBinding() {

			@Override
			protected int computeValue() {
				return internalLoadCalculation();
			}
		};
	}

    public BooleanBinding passengerPresentProperty() {
        return passengerPresent;
    }
	@Override
	public int getDamage() {
		return 100 - fitness;
	}

	@Override
	public void damage(int damage) {
		fitness = Math.max(0, fitness - damage);
	}

    @Override
    public int getLoadableSpace() {
        return getSize() - getUpgradeSpaceReduction();
    }

    @Override
    public double getTopSpeed() {
        return topSpeed;
    }
}