package ch.sahits.game.openpatrician.model.factory;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.ILandBridge;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.city.IChurch;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IFutureTowns;
import ch.sahits.game.openpatrician.model.city.ILoaner;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.LoanerList;
import ch.sahits.game.openpatrician.model.city.cityhall.CityHallList;
import ch.sahits.game.openpatrician.model.city.cityhall.IAcceptedAldermanTask;
import ch.sahits.game.openpatrician.model.city.cityhall.IBowmen;
import ch.sahits.game.openpatrician.model.city.cityhall.IBuildLandPassage;
import ch.sahits.game.openpatrician.model.city.cityhall.ICapturePirateNest;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityGuard;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHallNotice;
import ch.sahits.game.openpatrician.model.city.cityhall.ICrossbowmen;
import ch.sahits.game.openpatrician.model.city.cityhall.IAldermanOffice;
import ch.sahits.game.openpatrician.model.city.cityhall.IFoundNewSettlement;
import ch.sahits.game.openpatrician.model.city.cityhall.IHelpCity;
import ch.sahits.game.openpatrician.model.city.cityhall.IHuntPirate;
import ch.sahits.game.openpatrician.model.city.cityhall.IMusketeer;
import ch.sahits.game.openpatrician.model.city.cityhall.IPikemen;
import ch.sahits.game.openpatrician.model.city.cityhall.ITreasury;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.CapturePirateNest;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.CityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.Treasury;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.people.PeopleFactory;
import ch.sahits.game.openpatrician.model.people.impl.SeaPiratesState;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.util.CityUtilities;
import ch.sahits.game.openpatrician.util.MapSegmentImageFactory;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import javafx.collections.ObservableList;
import javafx.util.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Random;

/**
 * Factory to create all the dependent state model objects.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Nov 30, 2014
 */
@Service
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class StateFactory {
    private final Logger logger = LogManager.getLogger(getClass());
    private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
    private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");

    /**
     * Factor by which the price is multiplied for ware notices.
     */
    public static final double NOTICE_WARE_PRICE_FACTOR = 1.15;
    @Autowired
    private ApplicationContext context;
    @Autowired
    private CityUtilities citiesInteractionService;
    @Autowired
    private Random rnd;
    @Autowired
    private PeopleFactory peopleFactory;
    @Autowired
    private MapSegmentImageFactory segmentImageFactory;
    @Autowired
    private Date date;
    @Autowired
    private IMap map;
    @Autowired
    private SeaPiratesState pirateState;
    @Autowired
    private CityHallList cityHalls;
    @Autowired
    private LoanerList loaners;

    public IShipyard createShipYard(ICity city) {
        return (IShipyard) context.getBean("shipyardState", new Object[]{city});
    }
    public TavernState createTavernState(ICity city) {
        return (TavernState) context.getBean("tavernState", new Object[]{city});
    }

    public ILoaner createLoaner(ICity city) {
        ILoaner loaner = (ILoaner) context.getBean("loanerState", new Object[]{city});
        loaner.update();
        loaners.add(loaner);
        return loaner;
    }

    public ICityHall createCityHall(ICity city){
//        ArrayList<ICityHallNotice> notices = new ArrayList<>();
        ArrayList<ICitizen> candidates = new ArrayList<>();
        ITreasury treasury = new Treasury();
        treasury.cashProperty().setValue(rnd.nextInt(5000));
        treasury.currentPropertyTaxProperty().setValue(30);
        treasury.currentHeadTaxValueProperty().setValue(8);

        ArrayList<ICityGuard> guards = new ArrayList<>();
        ObservableList<ICityHallNotice> notices = citiesInteractionService.createNotices(city);
        final int population = city.getPopulationBinding().get();
        int nbCouncilmen = citiesInteractionService.getMaxNumberOfGuards(population);
        ArrayList<ICitizen> councilmen = new ArrayList<>();
        ICitizen mayor = null;
        for (ICitizen citizen : city.getCitizen()) {
            if (councilmen.size() < nbCouncilmen && citizen.getHometown().equals(city)) {
                final ESocialRank rank = citizen.getRank();
                if ((rank == ESocialRank.COUNCILMAN) ||
                    (rank == ESocialRank.PATRICIAN) ||
                    (rank == ESocialRank.MAYOR) ||
                    (rank == ESocialRank.ALDERMAN)) {
                    councilmen.add(citizen);
                    if (rank == ESocialRank.MAYOR) {
                        mayor = citizen;
                    }
                }
            }
        }
        if (mayor == null) {
            mayor = createCitizen(city, ESocialRank.MAYOR);
            if (councilmen.size() >= nbCouncilmen) {
                ICitizen citizen = councilmen.remove(councilmen.size()-1);
            }
            councilmen.add(mayor);
            city.getCitizen().add(mayor);
        }
        while (councilmen.size() < nbCouncilmen) {
            final ICitizen citizen = createCitizen(city, ESocialRank.COUNCILMAN);
            councilmen.add(citizen);
            city.getCitizen().add(citizen);
        }
        // Find the mayor
        candidates.add(mayor);
        int nbCandidates = Math.min(4, nbCouncilmen);
        while (candidates.size() < nbCandidates) {
            int index = rnd.nextInt(councilmen.size());
            ICitizen candidate = councilmen.get(index);
            if (!candidates.contains(candidate)) {
                candidates.add(candidate);
            }
        }

        // Guards
        int nbGuards = getNumberOfGuards(population);
        for (int i=0; i< nbGuards/2; i++) {
            guards.add(createPikeman());
        }
        while (guards.size() < nbGuards) {
            guards.add(createBowman());
        }
        CityHall cityHall = (CityHall) context.getBean("cityHall", new Object[]{notices, candidates, treasury, guards, city, councilmen});
        cityHall.setMayor(mayor);
        DateTime election = date.getCurrentDate().plusDays(rnd.nextInt(600));
        cityHall.setElectionDate(election);
        cityHall.setMaxNumberMilita(nbGuards);
        cityHalls.add(cityHall);
        return cityHall;
    }

//    private int getNumberOfConcilmen(int population) {
//        if (population < 1000) {
//            return 5;
//        }
//        double b = 3.8;
//        if (population < 20000) {
//            return  (int)Math.rint(population/760.0 + b);
//        } else {
//            return  (int)Math.rint(population/(760.0*2) + b);
//        }
//    }

    private int getNumberOfGuards(int population) {
        return population/100;
    }

    public ICitizen createCitizen(ICity hometown, ESocialRank rank) {
        String firstName = firstNameLoader.getRandomName();
        String lastName = lastNameLoader.getRandomName();
        return (ICitizen) context.getBean("citizen", new Object[]{lastName, firstName, rank, hometown});
    }
    public IPikemen createPikeman() {
        return context.getBean(IPikemen.class);
    }
    public IBowmen createBowman() {
        return context.getBean(IBowmen.class);
    }
    public ICrossbowmen createCrossbowman() {
        return context.getBean(ICrossbowmen.class);
    }
    public IMusketeer createMusketieer() {
        return context.getBean(IMusketeer.class);
    }
    public IAldermanOffice createAldermanOffice() {return context.getBean(IAldermanOffice.class);}

    /**
     * Create an alderman task to help a city. If no city with less than 2000 inhabitants is found absent will be returned.
     * @return
     */
    public Optional<IHelpCity> createHelpCityAldermanTask() {
         ArrayList<ICity> cities = new ArrayList<>(map.getCities());
        Collections.shuffle(cities);
        for (ICity city : cities) {
            final int population = city.getPopulationBinding().get();
            if (population < 2000) {
                int duration = (2000 - population)/ 3;
                final IHelpCity task = (IHelpCity) context.getBean("helpCity", new Object[]{duration, city});
                return Optional.of(task);
            }
        }
        return Optional.empty();
    }
    /**
     * Create an alderman task to help a city. If no city with less than 2000 inhabitants is found absent will be returned.
     * @return
     */
    public Optional<IHelpCity> createHelpCityAldermanTask(IAldermanOffice office) {
        List<IAcceptedAldermanTask> workedOnTasks = office.getWorkedOnTasks();
        List<ICity> helpTasks = new ArrayList<>();
        for (IAcceptedAldermanTask workedOnTask : workedOnTasks) {
            if (workedOnTask.getTask() instanceof IHelpCity) {
                helpTasks.add(((IHelpCity) workedOnTask.getTask()).getCity());
            }
        }
        ArrayList<ICity> cities = new ArrayList<>(map.getCities());
        Collections.shuffle(cities);
        for (ICity city : cities) {
            if (helpTasks.contains(city)) {
                continue;
            }
            final int population = city.getPopulationBinding().get();
            if (population < 2000) {
                int duration = (2000 - population)/ 3;
                final IHelpCity task = (IHelpCity) context.getBean("helpCity", new Object[]{duration, city});
                return Optional.of(task);
            }
        }
        return Optional.empty();
    }

    /**
     * Create an alderman task for founding a new city while ignoreing the ones that are worked on.
     * @param office
     * @return
     */
    public Optional<IFoundNewSettlement> createNewSettlementEledermanTask(IAldermanOffice office) {
        List<IAcceptedAldermanTask> workedOnTasks = office.getWorkedOnTasks();
        List<String> tasks = new ArrayList<>();
        for (IAcceptedAldermanTask workedOnTask : workedOnTasks) {
            if (workedOnTask.getTask() instanceof IFoundNewSettlement) {
                tasks.add(((IFoundNewSettlement) workedOnTask.getTask()).getName());
            }
        }
        ArrayList<IFutureTowns> cities = new ArrayList<>(map.getFutureTowns());
        Collections.shuffle(cities);
        for (IFutureTowns city : cities) {
            if (tasks.contains(city.getName())) {
                continue;
            }
            int duration = 720;
            final IFoundNewSettlement task = (IFoundNewSettlement) context.getBean("foundNewSettlement", duration, city.getLocation(), city.getName());
            return Optional.of(task);
        }
        return Optional.empty();
    }

    /**
     * Create an alderman task for building a land bridge while ignoring the once already worked on.
     * @param office
     * @return
     */
    public Optional<IBuildLandPassage> createNewLandBridgeAldermanTask(IAldermanOffice office) {
        List<IAcceptedAldermanTask> workedOnTasks = office.getWorkedOnTasks();
        List<Pair<ICity, ICity>> tasks = new ArrayList<>();
        for (IAcceptedAldermanTask workedOnTask : workedOnTasks) {
            if (workedOnTask.getTask() instanceof IBuildLandPassage) {
                final IBuildLandPassage concreteTask = (IBuildLandPassage) workedOnTask.getTask();
                Pair<ICity, ICity> pair = new Pair<>(concreteTask.getFromCity(), concreteTask.getToCity());
                tasks.add(pair);
            }
        }
        ArrayList<ILandBridge> landBridges = new ArrayList<>(map.getLandbridges());
        Collections.shuffle(landBridges);
        for (ILandBridge landBridge : landBridges) {
            Pair<ICity, ICity> pair = new Pair<>(landBridge.getFrom(), landBridge.getTo());
            if (tasks.contains(pair)) {
                continue;
            }
            int duration = 360;
            IBuildLandPassage newPassage = (IBuildLandPassage) context.getBean("buildLandPassage", duration, landBridge.getFrom(), landBridge.getTo());
            return Optional.of(newPassage);
        }
        return Optional.empty();
    }

    /**
     * Create an alderman task to hunt a pirate, while ignoring the ones that are already worked on.
     * @param office
     * @return
     */
    public Optional<IHuntPirate> createPirateHuntEledermanTask(IAldermanOffice office) {
        List<IAcceptedAldermanTask> workedOnTasks = office.getWorkedOnTasks();
        List<ISeaPirate> tasks = new ArrayList<>();
        for (IAcceptedAldermanTask workedOnTask : workedOnTasks) {
            if (workedOnTask.getTask() instanceof IHuntPirate) {
                tasks.add(((IHuntPirate)workedOnTask.getTask()).getPirate());
            }
        }
        List<ISeaPirate> pirates = new ArrayList<>(pirateState.getFreePirates());
        Collections.shuffle(pirates);
        for (ISeaPirate pirate : pirates) {
            if (tasks.contains(pirate)) {
                continue;
            }
            int duration = 60;
            IHuntPirate hunt = (IHuntPirate) context.getBean("huntPirate", duration, pirate);
            return Optional.of(hunt);
        }
        return Optional.empty();
    }

    /**
     * Create a new alderman task for destroing a pirats nest, while ignoring the ones that are already worked on.
     * @param office
     * @return
     */
    public Optional<ICapturePirateNest> createCapturePirateNestAldermanTask(IAldermanOffice office) {
        List<IAcceptedAldermanTask> workedOnTasks = office.getWorkedOnTasks();
        List<MapSegmentedImage> tasks = new ArrayList<>();
        for (IAcceptedAldermanTask workedOnTask : workedOnTasks) {
            if (workedOnTask.getTask() instanceof ICapturePirateNest) {
                tasks.add(((ICapturePirateNest)workedOnTask.getTask()).getPirateNestMap());
            }
        }
        MapSegmentedImage map = segmentImageFactory.getRandomPirateNest();
        if (tasks.contains(map)) {
            return Optional.empty();
        } else {
            int duration = 180;
            CapturePirateNest task = (CapturePirateNest) context.getBean("capturePirateNest", duration, map.getLocation(), map);
            return Optional.of(task);
        }
    }

    public IChurch createChurch() {
        return context.getBean(IChurch.class);
    }
}
