package ch.sahits.game.openpatrician.model.impl;

import ch.sahits.game.event.data.PeriodicalTimeDayUpdate;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.CityFactory;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.javafx.IJavaFXApplicationThreadExecution;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.LongProperty;
import javafx.beans.property.SimpleLongProperty;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.List;
import java.util.Set;

@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public final class Company implements ICompany {
	@XStreamOmitField
	private final Logger logger = LogManager.getLogger(Company.class);

	private final IPlayer owner;
    @Getter
	private ICity homeTown;
    @Getter
	private long companyValue;
	private LongProperty cash;
    @Autowired
	@XStreamOmitField
	private CityFactory cityFactory;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;
	@Autowired
	@XStreamOmitField
	private IJavaFXApplicationThreadExecution threadExecutor;


	public Company(IPlayer owner, ICity homeTown, long cash) {
		super();
		this.owner = owner;
		this.homeTown = homeTown;
		this.cash = new SimpleLongProperty(this, "cash", cash);
	}

    /**
     * Register the periodical update timer for cash changes.
     */
    @PostConstruct
    private void initialzeUpdates() {
		clientServerEventBus.register(this);
    }
	@PreDestroy
	private void unregister() {
		clientServerEventBus.unregister(this);
	}
	@Subscribe
	public void handelDailyUpdates(PeriodicalTimeDayUpdate event) {
		companyValue = cash.get();
		// Add value of all ships
		List<IShip> ships = owner.getFleet();
		for (IShip ship : ships) {
			companyValue += ship.getValue();
			Set<IWare> loaded = ship.getLoadedWares();
			for (IWare ware : loaded) {
				AmountablePrice<IWare> ap = ship.getWare(ware);
				companyValue+=ap.getAVGPrice()*ap.getAmount();
			}
		}

		// Add value of all buildings

		for (ICity city :  cityFactory.getCities()) {
            List<IBuilding> buildings = owner.findBuildings(city);
            for (IBuilding building : buildings) {
                companyValue += building.getValue();
                if (building instanceof ITradingOffice){
                    ITradingOffice office = (ITradingOffice) building;
                    for (IWare ware : EWare.values()) {
                        AmountablePrice<IWare> ap = office.getWare(ware);
                        companyValue+=ap.getAVGPrice()*ap.getAmount();
                    }
                }
            }

		}
	}

	@Override
	public long getCash() {
		return cash.get();
	}
	/**
	 * Add or subtract some cash
	 * @param diff amount of money that is transferred
	 */
	@Override
	public void updateCash(long diff){
		threadExecutor.execute(() -> {
			cash.set(cash.get() + diff);
		});
	}

    @Override
    public LongProperty cashProperty() {
        return cash;
    }
}
