package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import javafx.beans.binding.DoubleBinding;
import javafx.beans.binding.When;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import ch.sahits.game.openpatrician.model.city.ECityState;
import ch.sahits.game.openpatrician.model.city.IPopulationStructure;
/**
 * This is a very simple implementation of the price calculation algorithm.
 * It is based on the cosinus and ignores theproductionRate, population structure
 * and state of the city.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 26, 2011
 *
 */
@ClassCategory(EClassCategory.HANDLER)
final class SimplePriceCalculation implements IPriceCalculation {
	/**
	 * Compute the price for one item of the ware. Though the price is only handled in further
	 * computation as integer value, this method returns its value as double so this condition can be
	 * tested:<br>
	 * The higher the available amount (but still below the saturation value) the smaller (strictly) the returned
	 * value.<br>
	 * For different starting values:<br>
	 * If  the starting value b is below starting point a the whole curve of b is below (not necessary strictly)
	 * the curve of a. The curve is continues between 0 (inclusive) and infinity.<br>
	 * Compute the price based on the cosine. If nothing is available the max value
	 * is returned, if more than the saturation value is available the min value is
	 * returned. In between the price is defined by the cosine which produces a curve that
	 * is pretty shallow towards the two extremes.
 	 * @param min the lowest possible price (0 exclusive)
	 * @param max the highest possible price
	 * @param available amount of wares that are available in the market
	 * @param saturation amount of items at which the market is saturated (no further price change
	 * @param productionRate amount the city can produce within a week
	 * @param pop population structure that may have an influence on the calculated price
	 * @param state state of the city
	 * @return price for one item sold
	 */
	@Override
	public double computePrice(int min, int max, int available, int saturation,
			int productionRate, IPopulationStructure pop, ECityState state) {
		// FIXME: this implementation is not continuous and the two curves cross each other
		if (available==0) return max;
		if (available>=saturation) return min;
		// The cosine is defined on [0,PI] therefore we must stretch it to
		// saturation and then retrieve the value at the available amount
		double val =  Math.cos(available*Math.PI/saturation);
		// This gives us a value between [-1,1]
		val += 1;
		val *= 0.5;
		// Now between [0,1]
		// Stretch it
		val *= (max-min);
		// place it
		val += min;
//		val = val*(max-min)/2+min*2+min;
		return val;
	}
	/**
	 * Compute the price for one item of the ware. Though the price is only handled in further
	 * computation as integer value, this method returns its value as double so this condition can be
	 * tested:<br>
	 * The higher the available amount (but still below the saturation value) the smaller (strictly) the returned
	 * value.<br>
	 * For different starting values:<br>
	 * If  the starting value b is below starting point a the whole curve of b is below (not necessary strictly)
	 * the curve of a. The curve is continues between 0 (inclusive) and infinity.<br>
	 * Compute the price based on the cosine. If nothing is available the max value
	 * is returned, if more than the saturation value is available the min value is
	 * returned. In between the price is defined by the cosine which produces a curve that
	 * is pretty shallow towards the two extremes.
 	 * @param min the lowest possible price (0 exclusive)
	 * @param max the highest possible price
	 * @param available amount of wares that are available in the market
	 * @param saturation amount of items at which the market is saturated (no further price change
	 * @param productionRate amount the city can produce within a week
	 * @param pop population structure that may have an influence on the calculated price
	 * @param state state of the city
	 * @return price for one item sold as a DoubleBinding
	 */
	// TODO aho Nov 30, 2013: remove
	public DoubleBinding computePrice(ReadOnlyIntegerProperty min, ReadOnlyIntegerProperty max, IntegerProperty available, ReadOnlyIntegerProperty saturation,
			int productionRate, IPopulationStructure pop, ECityState state) {
		return (DoubleBinding) new When(available.isEqualTo(0))
					.then(max)
					.otherwise(new When(available.greaterThanOrEqualTo(saturation))
								.then(min)
								.otherwise(cosinus(available, saturation.get())
										           .add(1)
										           .multiply(0.5)
										           .multiply(max.subtract(min))
										           .add(min)
								)
					);
	}
	
	private DoubleBinding cosinus(final IntegerProperty available, final int saturation) {
		return new DoubleBinding() {
			{
				super.bind(available);
			}
			
			@Override
			protected double computeValue() {
				return Math.cos(available.getValue()*Math.PI/saturation);
			}
		};
	}
}
