package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.city.ICity;
import com.google.common.annotations.VisibleForTesting;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;

/**
 * Service for calculating amounts that might be bought for a certain price.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 24, 2014
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class WareAmountCalculator {

    // fixme: andi 12/25/14: if there is a discount factor for buying the amount will always be smaller than the amount that could be afforded

    /**
     * Calculate the amount of <code>ware</code> that can be bought
     * for at most <code>avgPrice</code>.
     * @param ware to be bought
     * @param city from which to buy
     * @param avgPrice for one item to be bought
     * @return amount that can be bought without surpassing the <code>avgPrice</code>.
     */
    public int calculateBuyAmount(IWare ware, ICity city, int avgPrice) {
        return calculateAmount(ware, city, avgPrice, true);

    }
    /**
     * Calculate the amount of <code>ware</code> that can be sold
     * for before dropping below <code>avgPrice</code>.
     * @param ware to be sold
     * @param city to which to sell
     * @param avgPrice for one item to be sold
     * @return amount that can be sold without dropping below <code>avgPrice</code>.
     */
    public int calculateSellAmount(IWare ware, ICity city, int avgPrice) {
        return calculateAmount(ware, city, avgPrice, false);

    }
    private int calculateAmount(IWare ware, ICity city, int avgPrice, boolean buying) {
        // Price for 100 item
        final SimpleIntegerProperty availableAmount = new SimpleIntegerProperty(city.getWare(ware).getAmount());
        int amount = 100;
        int price = calculateAvgPrice(ware, availableAmount, amount, buying);
        while (price < avgPrice) {
            amount += 100;
            price = calculateAvgPrice(ware, availableAmount, amount, buying);
        }
        if (price == avgPrice) {
            return amount;
        }
        int upperLimit = amount;
        int lowerLimit = amount - 100;
        return searchAvgPrice(ware, availableAmount, avgPrice, lowerLimit, upperLimit, buying);
    }

    @VisibleForTesting
    int searchAvgPrice(IWare ware, IntegerProperty availableAmount, int avgPrice, int lowerAmount, int upperAmount, boolean buying) {
        if (lowerAmount == upperAmount - 1) {
            return lowerAmount;
        }
        int amount = upperAmount - (upperAmount - lowerAmount)/2;
        int price = calculateAvgPrice(ware, availableAmount, amount, buying);
        if (price == avgPrice) {
            return amount;
        } else if (price > avgPrice) {
            // search in lower half
            return searchAvgPrice(ware, availableAmount, avgPrice, lowerAmount, amount, buying);
        } else {
            // search in upper half
            return searchAvgPrice(ware, availableAmount, avgPrice, amount, upperAmount, buying);
        }
    }

    private int calculateAvgPrice(IWare ware, IntegerProperty availableAmount ,final int amount, boolean buying) {
        IntegerBinding buyAmount = new IntegerBinding() {
            @Override
            protected int computeValue() {
                return amount;
            }
        };
        if (buying) {
            return ware.buyPrice(availableAmount, buyAmount);
        } else {
            return ware.sellPrice(availableAmount, buyAmount);
        }
    }
}
