package ch.sahits.game.openpatrician.model.util;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.collections.SortedMapRandomizedSameElements;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.CityHallList;
import ch.sahits.game.openpatrician.model.city.cityhall.ECityViolationPunishment;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHallNotice;
import ch.sahits.game.openpatrician.model.city.cityhall.ITreasury;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.CityHallNotice;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.factory.StateFactory;
import ch.sahits.game.openpatrician.model.people.IContractBroker;
import ch.sahits.game.openpatrician.model.people.PeopleFactory;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Range;
import javafx.beans.binding.IntegerBinding;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.geometry.Point2D;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;

/**
 * Utility class to privide services around the cities.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 27, 2013
 *
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class CityUtilities {
	@Autowired
	private IMap map;
	@Autowired
	private Random rnd;
    @Autowired
    private Date date;
    @Autowired
    private PeopleFactory peopleFactory;
    @Autowired
    private CityHallList cityHalls;

	private final Range<Double> range = Range.openClosed(0.0, 1.0);

	/**
	 * Find a city nearby. The search is limited by a factor (0,1]. Based on
	 * the <code>max</code> dimension of the map the radius is calculated with this
	 * factor.
	 * Repeated calls with the same arguments are not guaranteed to deliver the same result
	 * @param startCity city where to start
	 * @param radiusFactor factor for the radius calculation
	 * @param max maximal dimension of the map
	 * @param exclude potentially empty list of cities, that should be excluded from the search
	 * @return Optional city within the radius.
	 */
	public Optional<ICity> findNearbyCity(ICity startCity, double radiusFactor, int max, List<ICity> exclude) {
		List<ICity> cities = map.getCities();
		Preconditions.checkArgument(range.contains(radiusFactor), "Range factor must be in (0,1]");
		final double radius = max*radiusFactor;
		LinkedList<ICity> copy = new LinkedList<ICity>(cities);
		Collections.shuffle(copy);
		final Point2D p1 = startCity.getCoordinates();
		for (ICity city : copy) {
			if (!exclude.contains(city) && !city.equals(startCity)) {
				Point2D p2 = city.getCoordinates();
				double dist = calculateDistance(p1, p2);
				if (dist<=radius) {
					return Optional.of(city);
				}
			}
		}
		return Optional.empty();
	}
	/**
	 * Find a nearby city. If the radius is chosen to small the radius is increased successively.
	 * This method is guaranteed to deliver a result if there is a city that can be chosen.
	 * @param startCity city where to start
	 * @param radiusFactor factor for the radius calculation
	 * @param max maximal dimension of the map
	 * @param exclude potentially empty list of cities, that should be excluded from the search
	 * @return Optional city within the radius.
	 * @see #findNearbyCity(ICity, double, int, List)
	 */
	public ICity findNearbyCityRepeated(ICity startCity, double radiusFactor, int max, List<ICity> exclude) {
		Preconditions.checkArgument(checkCities(startCity,exclude), "No city is choosable");
		Optional<ICity> result = findNearbyCity(startCity, radiusFactor, max, exclude);
		while (!result.isPresent()) {
			radiusFactor = radiusFactor*1.01;
			result = findNearbyCity(startCity, radiusFactor, max, exclude);
		}
		return result.get();
	}
	/**
	 * Check if there is a city that can be choosen
	 * @param startCity
	 * @param exclude
	 * @return
	 */
	private boolean checkCities(ICity startCity, List<ICity> exclude) {

		List<ICity> cities = map.getCities();
		for (ICity city : cities) {
			if (city.equals(startCity)) {
				continue;
			}
			if (exclude.contains(city)) {
				continue;
			}
			return true;
		}
		return false;
	}
	/**
	 * Find the nearest city to a point
	 * @param p
	 * @return
	 */
	public ICity findNearestCity(Point2D p) {
		List<ICity> cities = map.getCities();
		ICity nearest = cities.get(0);
		double dist = calculateDistance(p, nearest.getCoordinates());
		for (int i=1; i<cities.size(); i++) {
			ICity city = cities.get(i);
			double d = calculateDistance(p, city.getCoordinates());
			if (d<dist) {
				dist = d;
				nearest = city;
			}
		}
		return nearest;
	}
	/**
	 * Find a random city that is not the excluded city.
	 * @param excludedCity city that cannot be chosen.
	 * @return
	 */
	public ICity findRandomCity(ICity excludedCity) {
        return findRandomCity(Arrays.asList(excludedCity));
	}
    /**
     * Find a random city that is not the excluded city.
     * @param excludedCity city that cannot be chosen.
     * @return
     */
    private ICity findRandomCity(List<ICity> excludedCity) {
        List<ICity> cities = Lists.newArrayList(map.getCities());
        cities.removeAll(excludedCity);
        int index = rnd.nextInt(cities.size());
        return cities.get(index);
    }
	private double calculateDistance(Point2D p1, Point2D p2) {
		double diffX = p1.getX()-p2.getX();
		double diffY = p1.getY()-p2.getY();
		return Math.sqrt(diffX*diffX+diffY*diffY);
	}

    /**
     * Find up to <code>numberOfCities</code> random cities that are not the excluded one.
     * @param exclude city that should be ignored
     * @param numberOfCities number of cities in the list. If the number is bigger than the max amount of cities on the
     *                       map, only that amount minus one will be used.
     * @return list of cities with the specified number.
     */
    public List<ICity> findRandomCities(ICity exclude, int numberOfCities) {
        int max = Math.min(map.getNumberCities() - 1, numberOfCities);
        List<ICity> selected = new ArrayList<>();
        selected.add(exclude);
        while (selected.size() < max + 1) {
            ICity next = findRandomCity(selected);
            selected.add(next);
        }
        selected.remove(exclude);
        return selected;
    }

    /**
     * Calculate the maximum of possible city guards base on the size of the population
     * @param population of the city
     * @return max number of possible guards
     */
    public int getMaxNumberOfGuards(int population) {
        if (population <= 1000) {
            return 5;
        }
        double b = 3.8;
        if (population <= 20000) {
            final double v = population / (2*(30 + b));
            return  (int)Math.rint((v + b)/10);
        } else {
            final double v = population / (2.65*(30 + b));
            return  (int)Math.rint((v + b)/10);
        }
    }

    public ObservableList<ICityHallNotice> createNotices(ICity targetCity) {
        ArrayList<ICityHallNotice> notices = new ArrayList<>();
        List<ICity> cities = findRandomCities(targetCity, 3);
        for (ICity iCity : cities) {
            CityState state = iCity.getCityState();
            Optional<IWare> ware = state.findMostNeededWare();
            if (ware.isPresent()) {
                int price = ware.get().sellPrice(iCity.getWare(ware.get()).amountProperty(), new IntegerBinding() {
                    @Override
                    protected int computeValue() {
                        return 5;
                    }
                });
                AmountablePrice<IWare> amountable = new AmountablePrice<>();
                int amount;
                if (ware.get().isBarrelSizedWare()) {
                    amount = rnd.nextInt(300) - 20;
                } else {
                    amount = rnd.nextInt(30) - 2;
                }
                int deliverInDays = rnd.nextInt(20) + 1;
                DateTime dueDate = date.getCurrentDate().plusDays(deliverInDays);
                amountable.add(amount, (int)(price * StateFactory.NOTICE_WARE_PRICE_FACTOR));
                IContractBroker broker = peopleFactory.createNewContractBroker(amountable, ware.get());
                ICityHallNotice notice = CityHallNotice.builder()
                        .contact(broker)
                        .destination(iCity)
                        .requiredWare(ware.get())
                        .wareAndAmount(amountable)
                        .dueDate(dueDate)
                        .build();
                notices.add(notice);
            }

        }
        return FXCollections.observableArrayList(notices);
    }

    public int getFine(ECityViolationPunishment punishment, ITreasury otherTreasury) {
        switch (punishment) {
            case SMALL_FINE:
                return (int) (otherTreasury.getCash()/100);
            case MEDIUM_FINE:
                return (int) (otherTreasury.getCash()/10);
            case LARGE_FINE:
                return (int) (otherTreasury.getCash()*0.9);
            case NONE:
                return 0;
            default:
                throw new IllegalStateException("The punishment "+punishment+" cannot be handled with a fine");
        }
    }

    /**
     * Create a sorted map of the candidates according to their reputation. As only {@link ch.sahits.game.openpatrician.model.IPlayer}
     * have a reputation any all {@link ch.sahits.game.openpatrician.model.ICitizen} have the same reputation of 0.
     * @param candidates
     * @return
     */
    public SortedMapRandomizedSameElements<Integer, ICitizen> getCandidateMap(List<ICitizen> candidates, ICity city) {
        List<ICitizen> citizens = new ArrayList<>();
        Map<Integer, List<ICitizen>> initialMap = new HashMap<>();
        initialMap.put(0,citizens);
        for (ICitizen citizen : candidates) {
            if (citizen instanceof IPlayer) {
                IReputation reputation = city.getReputation((IPlayer) citizen);
                int rep = reputation.getPopularity();
                if (initialMap.containsKey(rep)) {
                  List<ICitizen> list = initialMap.get(rep);
                    list.add(citizen);
                    initialMap.put(rep,list);
                } else {
                    ArrayList<ICitizen> list = new ArrayList<>();
                    list.add(citizen);
                    initialMap.put(rep,list);
                }
            } else {
                citizens.add(citizen);
            }
        }

        return new SortedMapRandomizedSameElements<>(initialMap);
    }

    /**
     * Check if the player is mayor of a city.
     * @param player which might be mayor
     * @param city where the player should be checkt
     * @return true if the player is mayor of the city.
     */
    public boolean isMayor(IPlayer player, ICity city) {
        for (ICityHall cityHall : cityHalls) {
            if (city.equals(cityHall.getCity())) {
                return player.equals(cityHall.getMayor());
            }
        }
        return false;
    }
}
