package ch.sahits.game.openpatrician.util;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.data.map.Map;
import ch.sahits.game.openpatrician.data.map.MapSegments;
import ch.sahits.game.openpatrician.data.map.PirateHideout;
import ch.sahits.game.openpatrician.data.map.PirateNest;
import ch.sahits.game.openpatrician.data.map.SunkenTreasure;
import ch.sahits.game.openpatrician.model.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.PirateMapSegmentImage;
import ch.sahits.game.openpatrician.model.TreasureMapSegmentImage;
import ch.sahits.game.openpatrician.model.city.CityFactory;
import ch.sahits.game.openpatrician.model.util.IMapSegmentImageCreator;
import ch.sahits.game.openpatrician.server.MapLocationDetectionModel;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

/**
 * This factory can only be used after it is initialized which is dependent on
 * the deserialisation of the map.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Nov 13, 2014
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class MapSegmentImageFactory {

    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private Random rnd;
    @Autowired
    private CityFactory cityFactory;
    @Autowired
    private IMapSegmentImageCreator imageCreator;
    @Autowired
    private MapLocationDetectionModel dataModel;


    @ListType(MapSegmentedImage.class)
    private List<MapSegmentedImage> pirateHideOuts;
    @ListType(MapSegmentedImage.class)
    private List<MapSegmentedImage> sunkShips;

    public void initializeImages() {
        Map map = cityFactory.getMap();
        pirateHideOuts = new ArrayList<>();
        sunkShips = new ArrayList<>();
        final MapSegments mapSegments = map.getMapSegments();
        imageCreator.loadMap(map.getImageName());
        for (PirateNest pirateNest : mapSegments.getPirateNest()) {
            try {
                imageCreator.createMapSegment(pirateNest, map.getImageName());
            } catch (IOException e) {
                logger.error("Failed to create pirate map segment.", e);
            }
//            pirateHideOuts.add(new PirateMapSegmentImage(pirateNest.getImgName(), new Point2D(pirateNest.getX(), pirateNest.getX()), pirateNest.getFleetStrength(), pirateNest.getName()));
        }
        for (SunkenTreasure sunkenTreasure : mapSegments.getSunkenTreasure()) {
            try {
                imageCreator.createMapSegment(sunkenTreasure, map.getImageName());
            } catch (IOException e) {
                logger.error("Failed to create pirate map segment.", e);
            }
            sunkShips.add(new TreasureMapSegmentImage(sunkenTreasure.getImgName(), new Point2D(sunkenTreasure.getX(), sunkenTreasure.getX()), sunkenTreasure.getTreasure()));
        }
        for (PirateHideout hideout : mapSegments.getPirateHideout()) {
            try {
                imageCreator.createMapSegment(hideout, map.getImageName());
            } catch (IOException e) {
                logger.error("Failed to create pirate map segment.", e);
            }
            pirateHideOuts.add(new PirateMapSegmentImage(hideout.getImgName(), new Point2D(hideout.getX(), hideout.getX()), 1, hideout.getName()));
        }
    }

    public MapSegmentedImage getRandomPirateNest() {
        MapSegmentedImage mapImage = getRandomImage(pirateHideOuts);
        return checkMapUnused(mapImage, pirateHideOuts);

    }

    private MapSegmentedImage checkMapUnused(MapSegmentedImage mapImage, List<MapSegmentedImage> segmentList) {
        do {
            if (!dataModel.containsSegment(mapImage)) {
                return mapImage;
            } else {
                mapImage = getRandomImage(segmentList);
            }
        } while (true);
    }

    public MapSegmentedImage getRandomTreasureNest() {
        MapSegmentedImage mapImage = getRandomImage(sunkShips);
        return checkMapUnused(mapImage, sunkShips);
    }
    private MapSegmentedImage getRandomImage(List<MapSegmentedImage> imgList) {
        MapSegmentedImage img = imgList.get(rnd.nextInt(imgList.size()));
        int availableSegment = rnd.nextInt(9);
        img.addSegment(availableSegment);
        return img;
    }
}
