package ch.sahits.game.openpatrician.model.city;

import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.city.impl.Contributions;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.geometry.Point2D;
import org.joda.time.DateTime;

import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Model of a city.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
public interface ICity {
	/**
	 * Retrieve the unique ID of the city.
	 * @return
	 */
	String getUniqueID();
	/**
	 * Retrieve the city type
	 * @return
	 */
	EKontorType getKontorType();
	/**
	 * Retrieve the city name
	 * @return
	 */
	String getName();

	/**
	 * Retrieve the current population of the city by class
	 * @param popclass
	 * @return
	 */
	int getPopulation(EPopulationClass popclass);
	/**
	 * Set the population by class
	 * @param population
	 * @param popClass
	 */
	void setPopulation(int population, EPopulationClass popClass);
	/**
	 * Add or remove ware from the city. If the amount is negative it will be removed.
	 * If the full amount is not available only the available will be moved and returned.
	 * The amount is in the ware specific size.
	 * @param ware to be moved
	 * @param amount of the ware that is moved
	 * @param player that moved the ware
	 * @return amount moved
	 */
	int move(IWare ware, int amount, ICitizen player);
	/**
	 * Retrieve the wares that are produced inefficiently
	 * @return
	 */
	IWare[] getIneffectiveProduction();
	/**
	 * Retrieve the wares that are produced efficiently
	 * @return
	 */
	IWare[] getEffectiveProduction();
	/**
	 * Retrieve a list of all buildings in the city. The list is immutable
	 * @return
	 */
	List<IBuilding> getBuildings();
	/**
	 * Add a new building to the city
	 * @param building
	 */
	void tearDown(IBuilding building);
	/**
	 * Remove a building from the list of buildings in the city
	 * @param building
	 */
	void build(IBuilding building);
	/**
	 * Retrieve the reputation of the player in the city
	 * @param player
	 * @return
	 */
	IReputation getReputation(IPlayer player);

	/**
	 * Add a player to the city
	 * @param player that is added to the city
	 * @param reputaion of that player in the city
	 * @param contributions contibutions of that player in the city.
	 */
	void moveIn(IPlayer player, IReputation reputaion, Contributions contributions);

	/**
	 * Check out what the player contributed to the ware
	 * @param player whose contribution is to be checked
	 * @param ware that is checked for contribution
	 * @return amount of the contribution may also be negative
	 */
	int getContribution(IPlayer player, IWare ware);
	/**
	 * Retrieve the missing wares along with their times
	 * @return
	 */
	Map<IWare, DateTime> getMissingWares();
	/**
	 * Retrieve the coordinates of the city
	 * @return
	 */
	Point2D getCoordinates();
	/**
	 * Retrieve all the players that have a office in this city.
	 * @return immutable list
	 */
	List<IPlayer> getResidentPlayers();
	/**
	 * Retrieve the city state.
	 * @return
	 */
	CityState getCityState(); // TODO aho Feb 3, 2013: create interface
	void setCityState(CityState state);
	/**
	 * Retrieve the amount of ware in the city
	 * @param ware
	 * @return
	 */
	AmountablePrice<IWare> getWare(IWare ware);
	/**
	 * Retrieve the population count for a class
	 * @param popclass population class
	 * @return
	 */
	IntegerProperty getPopulationProperty(EPopulationClass popclass);
	/**
	 * Retrieve the binding of the city's total population.
	 * @return
	 */
	IntegerBinding getPopulationBinding();

    /**
     * Retrieve the list of citizen in the city. This list contains various player as well as some instances used by the
     * AI to govern the city.
     * @return
     */
    List<ICitizen> getCitizen();

    /**
     * Find all matching buildings in a city.
     * @param buildingClass class of the building
     * @param owner search can be restricted to a player.
     * @return List of buildings of type T
     */
    <T extends IBuilding> List<T> findBuilding(Class<T> buildingClass, Optional<IPlayer> owner);

	/**
	 * Retrieve the non locale sepecific name template.
	 * @return
	 */
	String getNameTemplate();

}
