package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.model.city.IBuildingProduction;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.util.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.util.spring.DependentValue;
import com.google.common.annotations.VisibleForTesting;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import lombok.Setter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.TreeMap;

/**
 * This class encapsulates state information used by CityEngine about one city.
 * This class i player private since it should not be used by any other class than CityEngine.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 29, 2011
 *
 */
@Component
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class CityState {
	@XStreamOmitField
    private final Logger logger = LogManager.getLogger(getClass());

	@Autowired
	@XStreamOmitField
	private Random rnd;
	@DependentValue("ware.surplus.treshold")
	private double surplusThreshold;

	/** Reference to the city model that is driven by this engine */
    @Getter
	private ICity city;
	/** Map of consumed wares, since only whole units can be consumed we need to store the fractions */
	@MapType(key=IWare.class, value=Double.class)
	private Map<IWare, Double> consumed = new HashMap<>();
	/** Tavern state for this city */
    @Getter
	private TavernState tavernState = null;
	/** Ship yard state fos the city */
    @Getter
	private IShipyard shipyardState = null;
	/** Provider for the consume numbers */
	@Autowired
	private PopulationConsume consume;
	@Autowired
	@XStreamOmitField
	private IBuildingProduction buildingProduction;

	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;
	@Getter
    @Setter
    @Autowired
    private CityWall cityWall;

	public CityState(ICity city, IShipyard shipard, TavernState tavern) {
		this.city = city;
		this.shipyardState = shipard;
		this.tavernState = tavern;
	}
	@PostConstruct
	private void initCity() {
		EWare[] wares = EWare.values();
		for (EWare ware : wares) {
			replaceConsumedAmount(ware, 0.0);
		}
		if (city.getCityState() == null) {
			city.setCityState(this);
		}
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			logger.warn("Failed to initialize DependentValue annotated fields");
		}
	}
	/**
	 * Get the consumed entries as a set.
	 * @return
     */
	public Set<Entry<IWare, Double>> consumedEntries() {
		return consumed.entrySet();
	}

	/**
	 * Set the amount of the ware that is consumed. If there is already an amount
	 * stored for the ware it is overidden.
	 * @param ware to be consumed
	 * @param amount to be consumed
     */
	public void replaceConsumedAmount(IWare ware, double amount) {
		consumed.put(ware, amount);// does this produce ConcurrentModExc?
	}

	/**
	 * Retrieve the amount that is consumed so far.
	 * @param ware which should be checked for consumption.
	 * @return consumed amount.
     */
	public double getConsumedAmount(EWare ware) {
		return consumed.get(ware);
	}


	/**
	 * Find the ware which is consumed the most and which is missing for the longest time.
	 * @return
	 */
	public Optional<IWare> findMostNeededWare() {
		Map<IWare, DateTime> missing = city.getMissingWares();
		DateTime earliest = null;
		IWare result = null;
		for (Entry<IWare, DateTime> entry : missing.entrySet()) {
			if (earliest == null ||
					entry.getValue().isBefore(earliest)) {
				earliest = entry.getValue();
				result = entry.getKey();
			}
		}
        if (result == null) { // All wares are availalbe => Look which one will run out first
            Map<IWare, Integer> available = getSortedWareAvailabilityMap();
            IWare wareRunninOutFirst = null;
            int runsOutInDays = Integer.MAX_VALUE;
            for (Entry<IWare, Integer> entry : available.entrySet()) {
                IWare ware = entry.getKey();
                double consumptionPerWeek = consume.getWeeklyConsumption(ware, city);
                double productionPerWeek = buildingProduction.getTotalProduction(ware, getCity());
                double dailyConsumtion = (productionPerWeek - consumptionPerWeek)/7;
                if (dailyConsumtion >= 0) {
//                    System.out.println("DailyConsumtion of " + ware.name() + " is " + dailyConsumtion);
                    continue; // more ware is produced than consumed
                }
                int days = -(int)(entry.getValue()/dailyConsumtion);
//                System.out.println("DailyConsumption: " + dailyConsumtion + ", runsOutIn " + days);
                if (days < runsOutInDays) {
                    runsOutInDays = days;
                    wareRunninOutFirst = ware;
                }
            }
            result = wareRunninOutFirst;
        }

		return Optional.ofNullable(result);
	}
	/**
	 * Find the ware that the city sells near the minimal price.
	 * @return
	 */
	public Optional<IWare> findWareWithMostSurplus() {
		ArrayList<IWare> surplus = new ArrayList<IWare>();
		for (IWare ware : EWare.values()) {
			AmountablePrice<IWare> amount = city.getWare(ware);
			double actualPrice = amount.getAVGPrice();
			int minPrice = ware.getMinValueSell();
			double val = minPrice/actualPrice - 1;
			if (val<=surplusThreshold) {
				surplus.add(ware);
			}
		}
		if (surplus.isEmpty()) {
			return Optional.empty();
		} else {
			return Optional.of(surplus.get(rnd.nextInt(surplus.size())));
		}
	}

    /**
     * Find the wares and order them in the amount of their availability. If there
     * is a ware with the same availability, only one is added to the map.
     * @return
     */
    @VisibleForTesting
    Map<IWare, Integer> getSortedWareAvailabilityMap() {
        Comparator<IWare> comparator = new Comparator<IWare>() {
            @Override
            public int compare(IWare ware1, IWare ware2) {
                return city.getWare(ware1).getAmount() - city.getWare(ware2).getAmount();
            }
        };
        TreeMap<IWare, Integer> map = new TreeMap<>(comparator);
        for (IWare ware : EWare.values()) {
            map.put(ware, city.getWare(ware).getAmount());
        }
        map.size(); // this seems to be required to ensure the proper size is retrieved.
        return map;
    }
}
