package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.people.IWeaponsDealer;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.openpatrician.util.TransferUtil;
import ch.sahits.game.openpatrician.util.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.util.spring.DependentValue;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
/**
 * Implementation of the weapons dealer.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 25, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class WeaponsDealerState implements IWeaponsDealer {
	@XStreamOmitField
	private final Logger logger = LogManager.getLogger(getClass());
	@Autowired
    @XStreamOmitField
	private Random rnd;
	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;
	@DependentValue("tavern.weapons.dealer.sellvalue.factor")
	private double sellValue;
    @Autowired
    @XStreamOmitField
    private TransferUtil transferUtil;
    @MapType(key = IWeapon.class, value = IntegerProperty.class)
	private Map<IWeapon, IntegerProperty> weaponStore = new HashMap<>();
	@PostConstruct
	public void init() {
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			logger.warn("Failed to initialize DependentValue annotated fields");
		}
		// TODO aho Jan 25, 2013: make the amount dependent on something
		addNewWare(EWeapon.SWORD, rnd.nextInt(5));
		addNewWare(EWeapon.BOW, rnd.nextInt(5));
		addNewWare(EWeapon.CROSSBOW, rnd.nextInt(5));
		addNewWare(EWeapon.MUSKET, rnd.nextInt(5));
	}




	/**
	 * Add a new ware to the wares list
	 * @param weapon to be stored
     * @param amount to be stored
	 */
	private void addNewWare(IWeapon weapon, int amount){
		weaponStore.put(weapon, new SimpleIntegerProperty(amount));
	}

	/**
	 * Move ware into this holding. The player may be null and is not used in this
	 * base implementation, but subclasses may be interested for statistical reasons.
	 * This method is thread save.
	 * @param weapon to be moved
	 * @param amount of the ware that is moved
	 * @return the effective amount that was moved. The amount may be positive if something was added, negative if
	 * the ware was removed from the holding or zero if nothing was moved.
	 */
	@Override
	public int move(IWeapon weapon, int amount) {
		synchronized (weapon) { // There is still a problem with networked solution where the model of the city is replicated.
			int amounted = getWeapon(weapon);
			if (amount<0 && containsWare(weapon) &&  -amount>amounted){ // remove more than is available
				amount = -amounted; // remove everything
			}
			if (amount<0 && !containsWare(weapon)){
				amount = 0;
			}
            weaponStore.get(weapon).set(amounted+amount);
			return amount;
		}
	}
	/**
	 * Check if there is an amountable object for this ware
	 * @param weapons to be checked
	 * @return
	 */
	protected final boolean containsWare(IWeapon weapons) {
		return weaponStore.containsKey(weapons) && weaponStore.get(weapons).get() > 0;
	}

    /**
     * Retrieve the amount of the weapon that belongs to the dealer.
     * @param weapon type of weapon
     * @return amount available at the time of the call of the method.
     */
	private int getWeapon(IWeapon weapon) {
		return weaponStore.get(weapon).get();
	}


    @Override
    public IntegerProperty amountAvailableProperty(EWeapon weapon) {
        return weaponStore.get(weapon);
    }
}
