package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.BaseAmountable;
import ch.sahits.game.openpatrician.model.IAmountable;
import javafx.beans.binding.NumberBinding;
import javafx.beans.binding.When;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
/**
 * This class defines the amounted price of an {@link IWare} object.
 * Each object in the real world has hits unique price. Since no instances of the
 * objects are used but only one reference the price information gets lost. The
 * amountable price tracks the average price of all amounted objects
 * @param <T> {@link IAmountable} implementation that is is collected hereby called item
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 20, 2011
 * @param <T>
 *
 */
@Prototype
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class AmountablePrice<T extends IAmountable> extends BaseAmountable<T> {
	/** Total price of all amounted items */
	private final DoubleProperty sum = new SimpleDoubleProperty(this, "sum", 0);

    public AmountablePrice() {
    }

	/**
	 * Constructor initalizing the amountablePrice with an amount and average price.
	 * This constructor is mainly inteded for testing to avoid the hassle with the add method.
	 * @param amount
	 * @param totalPrice
     */
	public AmountablePrice(int amount, double totalPrice) {
		this.amount.setValue(amount);
		this.sum.setValue(totalPrice);
	}

    /**
	 * Retrieve the average price of one item
	 * @return
	 */
	public int getAVGPrice(){
		return (int)Math.rint(sum.doubleValue()/amount.get());
	}
	
	public NumberBinding avgPriceProperty() {
		return new When(sum.isEqualTo(0)
				        .or(amount.isEqualTo(0)))
		          .then(0)
		          .otherwise(sum.divide(amount));
	}
	
	
	/**
	 * Add a number items
	 * @param amount number of the items to be added
	 * @param avgPrice average price of one item that is added
	 */
	public void add(final int amount, final int avgPrice){
		fxTheadExecution.execute(() -> {
			int localAmount = amount;
			if (localAmount<0){
				localAmount = Math.abs(localAmount);
			}
			int current = this.amount.get();
			this.amount.set(current+localAmount);
			long sum = this.sum.longValue();
			this.sum.set(sum + localAmount*avgPrice);
		});
	}
	/**
	 * Remove a number of items
	 * @param amount of items to be removed
	 */
	public void remove (final int amount){

		fxTheadExecution.execute(() -> {
			int localAmount = amount;
			if (localAmount<0){
				localAmount = Math.abs(localAmount);
			}
			if (localAmount==Integer.MIN_VALUE){
				localAmount = Integer.MAX_VALUE;
			}
			if (this.amount.get()<localAmount){
				this.amount.set(0);
				sum.set(0);
			} else {
				long sum = this.sum.longValue();
				this.sum.set(sum - localAmount*getAVGPrice()); // sum must be updated first
				int current = this.amount.get();
				this.amount.set(current - localAmount);
			}
			if (this.amount.get()==0){
				sum.set(0);
			}
		});

	}
	
	@Override
	public void reset() {
		fxTheadExecution.execute(() -> {
			this.sum.set(0);
			super.reset();
		});

	}

	public String toString(){
		return amount.get()+"@"+getAVGPrice();
	}

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }

        AmountablePrice<?> that = (AmountablePrice<?>) o;

        return sum != null ? sum.get() == that.sum.get() : that.sum == null;

    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 31 * result + (sum != null ? sum.hashCode() : 0);
        return result;
    }

    /**
     * Get the total cum of the amount.
     * @return
     */
    public double getSum() {
        return sum.get();
    }

}
