package ch.sahits.game.openpatrician.model.ship;

import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.IntegerProperty;

import java.util.Optional;

/**
 * Model of a ship
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jun 15, 2011
 *
 */
public interface IShip extends INavigableVessel {

	/**
	 * Check if the ship can be upgraded
	 * @return
	 */
	boolean isUpgradable();
	/**
	 * Upgrade the ship if it is upgradable
	 */
	void upgrade();

	/**
	 * Set the ships name
	 * @param name
	 */
	void setName(String name);

	/**
	 * Repair the damage
	 */
	void repair();
	/**
	 * Retrieve the reduction of the load capacity based on the upgrade level.
	 * The reduction is in barrel that reduces the initial size
	 * @return reduction in barrels
	 */
	int getUpgradeSpaceReduction();

	/**
	 * check if weapons are on board
	 * @return
	 */
	boolean hasWeapons();
	/**
	 * Retrieve the current value of the ship without its cargo
	 * @return
	 */
	int getValue();

	/**
	 * Set the number of sailors.
	 * @param nbSailors .
	 */
	void setNumberOfSailors(int nbSailors);

    IntegerProperty numberOfSailorsProperty();
	/**
	 * Retrieve the minimum of sailors needed to operate the ship.
	 * @return
	 */
	int getMinNumberOfSailors();
	/**
	 * Retrieve the maximum of sailors that can be put on the ship.
	 * This amount depends on the ship type as well as the amount of load.
	 * @return
	 */
	int getMaxNumberOfSailors();

	/**
	 * Set the captain on the ship
	 * @param captain
	 */
	void setCaptain(ICaptain captain);
	/**
	 * Move weapons in or out.
	 * @param weaponType SWORD | BOW | CROSSBOW | MUSKET
	 * @param amount
	 */
	void move(IWeapon weaponType, int amount);

	/**
	 * Move a passenger to the ship. The ship may only contain one passanger.
	 * @param passanger
	 */
	void addPassenger(IPerson passanger);
	/**
	 * Passenger leave ship.
	 */
	void leavePassanger();

    /**
     * Retrieve the passenger on the ship.
     * @return optional wrapped around the passenger.
     */
    Optional<IPerson> getPassenger();
	/**
	 * Retrieve the type of the ship.
	 * @return
	 */
	EShipType getShipType();
	/**
	 * Define the owner of the ship
	 * @param player
	 */
	void setOwner(IShipOwner player);
	/**
	 * Retrieve the upgrade level of the ship.
	 * @return
	 */
	EShipUpgrade getShipUpgradeLevel();
	
	IShipWeaponsLocation getShipWeaponsLocation();

	/**
	 * Indicate if the ship is available. A ship may be unavailable during repair or upgrade.
	 * @return
	 */
	boolean isAvailable();

	/**
	 * Set the available flag on the ship.
	 * @param available
	 */
	void setAvailable(boolean available);

	int getOccupiedSpace();
	void setOccupiedSpace(int space);

	/**
	 * Binding indicating if a passenger is present.
	 * @return
     */
	BooleanBinding passengerPresentProperty();

	/**
	 * Set the ships weapons location.
	 * @param shipWeaponsLocation
     */
	void setShipWeaponsLocation(IShipWeaponsLocation shipWeaponsLocation);

	/**
	 * Retrieve the distance in km the ship has to travel under normal conditions, for it's health to be reduced by one.
	 * @return number of km to be traveled.
     */
	int getDistanceInKmForOneHealthPointReduction();
}
