package ch.sahits.game.openpatrician.model.building.impl;

import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ELevel;
import ch.sahits.game.openpatrician.model.building.IStorage;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWarehouse;
import ch.sahits.game.openpatrician.model.city.ICity;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.ObjectBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Optional;

@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class Storage implements IStorage {
	@Autowired
	@Qualifier("timerEventBus")
	@XStreamOmitField
	private AsyncEventBus timerEventBus;
	@Value("${storage.guard.cost.per.day}")
	private int guardCostPerDay = 10;
	@Value("${storage.rent.per.day}")
	private int rentCostPerDayAndBale = 20;

	private final IPlayer owner;
	private final ICity city;

	private IntegerProperty rentOutStorage;
	private IntegerProperty nbGuards;
	public Storage(IPlayer owner, ICity city) {
		super();
		this.owner = owner;
		this.city = city;
		rentOutStorage = new SimpleIntegerProperty(this, "rentOutStorage", 0);
		nbGuards = new SimpleIntegerProperty(this, "nbGuards", 0);
	}
	@PostConstruct
	private void init() {
		timerEventBus.register(this);
	}
	@PreDestroy
	public void destroy() {
		timerEventBus.unregister(this);
	}
	/**
	 * Retrieve the number of warehouses
	 * @return
	 */
	private int getNbWareHouses(){
		return owner.findBuildings(city, IWarehouse.class).size();
	}

	@Override
	public IntegerProperty rentOutStorageProperty() {
		return rentOutStorage;
	}

	@Override
	public void updateRendedSpace(int nbBarrels) {
		if (-nbBarrels>rentOutStorage.get()){
			rentOutStorage.set(0);
		} else {
			int newValue = rentOutStorage.get() + nbBarrels;
			rentOutStorage.set(newValue);
		}

	}

	@Override
	public IntegerBinding costsPerDayBinding() {
		Optional<ITradingOffice> optOffice = owner.findTradingOffice(city);
		Preconditions.checkArgument(optOffice.isPresent(), "Trading office must be present when there is Storage");
		final ITradingOffice office = optOffice.get();
		IntegerBinding additionalCosts = new IntegerBinding() {
			{
				super.bind(office.capacityProperty(), office.storedAmountBinding());
			}
			
			@Override
			protected int computeValue() {
				int additional = office.capacityProperty().subtract(office.storedAmountBinding()).intValue();
				if (additional>=0)
				{
					return 0; // empty space
				}
				return (-additional/10+1)* rentCostPerDayAndBale;
			}
		};
		return additionalCosts;
	}

	@Override
	public IntegerProperty numberGuardsProperty() {
		return nbGuards;
	}

	@Override
	public IntegerBinding guardCostsPerDayBinding() {
		return nbGuards.multiply(guardCostPerDay);
	}

	@Override
	public ObjectBinding<ELevel> securityLevelBinding() {
		return new ObjectBinding<ELevel>() {
			{
				super.bind(nbGuards); // TODO aho Dec 8, 2013: also bind on the number of warehouses
			}

			@Override
			protected ELevel computeValue() {
				return internalSecurityLevelCalculation();
			}
		};
	}
	private ELevel internalSecurityLevelCalculation() {
		double nb = getNbWareHouses();
		double factor =nbGuards.get()/(nb+1);
		if (factor>=1) {
			return ELevel.MAXIMUM;
		}
		if (factor>=0.66) {
			return ELevel.HIGH;
		}
		if (factor<=0.33) {
			return ELevel.LOW;
		}
		return ELevel.MEDIUM;
	}
	@Override
	public void updateGuardsNumber(int update) {
		if (-update>nbGuards.get()){ // reduce by more than are available
			nbGuards.set(0);
		} else {
			int newValue = nbGuards.get() + update;
			nbGuards.set(newValue);
		}

	}
	@Subscribe
	public void handleDailyUpdate(ClockTickDayChange event) {
		int costs = costsPerDayBinding().add(guardCostsPerDayBinding()).intValue();
		owner.getCompany().updateCash(-costs);
	}

}
