package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.datastructure.GenericPair;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.util.MapInitializedBean;
import com.google.common.collect.ImmutableList;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;

/**
 * State common to all cities.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 1, 2013
 *
 */
@Component
@Lazy
@DependentInitialisation(MapInitializedBean.class)
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class CitiesState {
	@Autowired
	private IMap map;
	@Autowired
	@XStreamOmitField
	private Random rnd;


	/** List of all the cities coupled with their state to be handeld by this engine */
	/* Use of  ConcurrentHashMap because later in the game cities might be added and we want
	 * to prevent ConcurrentModificationExceptions if we are currently iterating when the
	 * new city is added.
	 * Use map to retrieve city.
	 */
	@MapType(key=ICity.class, value = CityState.class)
	private ConcurrentHashMap<ICity, CityState> cities = new ConcurrentHashMap<>();
	public void init() {
		for (ICity city : map.getCities()) {
			if (cities.get(city) == null) {
				addCity(city, city.getCityState());
			}
		}
	}

	/**
	 * Add a new city.
	 * @param city
     */
	public void addCity(ICity city, CityState cityState) {
		cities.put(city, cityState);
	}

	/**
	 * Find a possible city that has needed wares. Repeated calls to this method are not garanteed to
	 * return the same city.
	 * @return
	 */
	public Optional<GenericPair<ICity, IWare>> findCityWithMostNeededWare() {
		List<GenericPair<ICity, IWare>> all = new ArrayList<>();
		for (CityState state : cities.values()) {
			Optional<IWare> ware = state.findMostNeededWare();
			if (ware.isPresent()) {
				all.add(new GenericPair<>(state.getCity(), ware.get()));
			}
		}
		if (all.isEmpty()) {
			return Optional.empty();
		} else {
			return Optional.of(all.get(rnd.nextInt(all.size())));
		}
	}

	/**
	 * Find a possible city that has needed wares. Repeated calls to this method are not garanteed to
	 * return the same city.
	 * @return
	 */
	public Optional<GenericPair<ICity, IWare>> findCityWithSurplusWare() {
		List<GenericPair<ICity, IWare>> all = new ArrayList<GenericPair<ICity,IWare>>();
		for (CityState state : cities.values()) {
			Optional<IWare> ware = state.findWareWithMostSurplus();
			if (ware.isPresent()) {
				all.add(new GenericPair<ICity, IWare>(state.getCity(), ware.get()));
			}
		}
		if (all.isEmpty()) {
			return Optional.empty();
		} else {
			return Optional.of(all.get(rnd.nextInt(all.size())));
		}
	}

	public List<CityState> getCityEngineStates(){
		return ImmutableList.copyOf(cities.values());
	}
}
