package ch.sahits.game.openpatrician.model.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.annotation.MultimapType;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.impl.PersonalData;
import ch.sahits.game.openpatrician.model.player.IAICaptainHireStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIConstructionSelectionStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIEventDecisionStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIGuildJoinStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIShipRepairStrategyType;
import ch.sahits.game.openpatrician.model.player.IAITakeLoanStrategyType;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import com.google.common.collect.LinkedHashMultimap;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import lombok.Setter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Implementation of an artificial intelligence player.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 17, 2011
 *
 */
@Component("aiPlayer")
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class AIPlayer extends Player implements IAIPlayer {
	@XStreamOmitField
	private static final Logger LOGGER = LogManager.getLogger(AIPlayer.class);

	@MultimapType(key = INavigableVessel.class, value = ITradeStep.class)
	private LinkedHashMultimap<INavigableVessel, ITradeStep> tradeSteps = LinkedHashMultimap.create();
	@MapType(key = INavigableVessel.class, value = ITradeMissionData.class)
	private Map<INavigableVessel, ITradeMissionData> tradeMissions = new HashMap<>();
	/**
	 * Waiting status of the trade for a ship. This is relevant for loading a game where
	 * certain ships are waiting for a trade step to finish (e.g. traveling or repairing).
     */
	@MapType(key = INavigableVessel.class, value = Boolean.class)
	private Map<INavigableVessel, Boolean> waitingStatus = new HashMap<>();
    @MapType(key = INavigableVessel.class, value = IAITradeStrategyType.class)
	private Map<INavigableVessel, IAITradeStrategyType> tradeStrategyType = new HashMap<>();
	@Getter
	@Setter
	private IAIShipRepairStrategyType shipRepairStrategyType;
	@Setter
	@Getter
	private IAIConstructionSelectionStrategyType constructionSelectionType;
	@Setter
	@Getter
	private IAICaptainHireStrategyType captainHireStrategyType;
	@Getter
	@Setter
	private IAIGuildJoinStrategyType guildJoinStrategyType;
	@Getter
	@Setter
	private IAITakeLoanStrategyType takeLoanStrategyType;
	@Getter
	@Setter
	private IAIEventDecisionStrategyType eventDecitionStrategyType;
	@Autowired
	private IProductionConsumptionKnowledge knowledge;

	private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
	private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");

	public AIPlayer(ICity homeTown,DateTime birthDate, ICompany company) {
		super(homeTown, getPersonalData(homeTown, birthDate),ESocialRank.BARGAINER, company);
	}
	// TODO ahotz 19.05.2016: update the knowledge with exact numbers for the home city

	private static PersonalData getPersonalData(ICity homeTown, DateTime birthDate) {
		return new PersonalData(getRandomFirstName(), getRandomLastName(), true, homeTown, birthDate);
	}

	private static String getRandomFirstName(){
		return firstNameLoader.getRandomName();
	}
	private static String getRandomLastName(){
		return lastNameLoader.getRandomName();
	}

	@Override
	public IProductionConsumptionKnowledge getProductionAndConsumptionKnowledge() {
		return knowledge;
	}

	@Override
	public ITradeStep getNextTradeStep(INavigableVessel vessel) {
		Collection<ITradeStep> steps = tradeSteps.get(vessel);
		ITradeStep first = steps.iterator().next();
		tradeSteps.remove(vessel, first);
		return first;
	}


	@Override
	public boolean hasMoreTradeSteps(INavigableVessel vessel) {
		return tradeSteps.get(vessel) != null && !tradeSteps.get(vessel).isEmpty();
	}

	@Override
	public void addTradeStep(ITradeStep tradeStep, INavigableVessel vessel) {
		tradeSteps.put(vessel, tradeStep);
	}

	@Override
	public void injectTradeStep(ITradeStep tradeStep, INavigableVessel vessel) {
		List<ITradeStep> existingSteps = new ArrayList(tradeSteps.get(vessel));
		tradeSteps.removeAll(vessel);
		tradeSteps.put(vessel, tradeStep);
		for (ITradeStep existingStep : existingSteps) {
			tradeSteps.put(vessel, existingStep);
		}
	}

	@Override
	public ITradeMissionData getTradeMission(INavigableVessel vessel) {
		return tradeMissions.get(vessel);
	}

	@Override
	public void setTradeMission(INavigableVessel vessel, ITradeMissionData tradeMission) {
		if (tradeMission != null) {
			tradeMissions.put(vessel, tradeMission);
		} else {
			tradeMissions.remove(vessel);
		}
	}

	@Override
	public boolean waitingForTradeStepToFinish(INavigableVessel vessel) {
		if (!isInitialized(vessel)) {
			waitingStatus.put(vessel, false);
		}
		return waitingStatus.get(vessel);
	}

	@Override
	public void updateTradeWaitingStatus(INavigableVessel vessel, boolean wait) {
		waitingStatus.put(vessel, wait);
	}
	@Override
	public boolean isInitialized(INavigableVessel vessel) {
		return waitingStatus.containsKey(vessel);
	}

	/**
	 * Set the trade strategy type on the level of a vessel.
	 * @param vessel
	 * @param type
     */
	@Override
	public void setTradeStrategyType(INavigableVessel vessel, IAITradeStrategyType type) {
		LOGGER.debug("Set trade strategy {} for vessel {}", type, vessel.getName());
		tradeStrategyType.put(vessel, type);
		tradeSteps.removeAll(vessel);
	}

	@Override
	public IAITradeStrategyType getTradeStrategyType(INavigableVessel vessel) {
		return tradeStrategyType.get(vessel);
	}
}
