package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import com.google.common.collect.Lists;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * Containing the states of the captains
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 2, 2013
 *
 */
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class CaptainsState {
	@Autowired
	@XStreamOmitField
	private Random rnd;

    @Value("${tavern.captain.presence.max}")
    @Getter
    private int maxDaysPresent;
    @Autowired
	private IMap map;

	@Autowired
	private Date date;
	@ListType(CaptainTownMapping.class)
	private List<CaptainTownMapping> captainMapping;
	@ListType(ICaptain.class)
	private List<ICaptain> captains;

	public CaptainsState() {
		captainMapping = Lists.newArrayList();
		captains = Lists.newArrayList();
	}

	public void addCaptain(ICity city, DateTime arrival, ICaptain captain) {
		assignCaptainToCity(captain, city, arrival);
		captains.add(captain);
	}

	private ICity getRandomCity() {
		List<ICity> cities = map.getCities();
		return cities.get(rnd.nextInt(cities.size()));
	}

	/**
	 * Assign the captain to a new town. The arrival there is delayed due to traveling time.
	 * @param city current town that the captain is leaving
	 * @param captain which travels to a new town.
	 */
	public void assignCaptainToCity(ICaptain captain, ICity city, DateTime arrivalDate) {
		captainMapping.add(new CaptainTownMapping(city, arrivalDate, captain));
	}

	/**
	 * Return a captain for a town if there is one. There may be more than one.
	 * @param city for which the captain is retrieved
	 * @return absent if no captain is in the town.
	 */
	public Optional<ICaptain> getCaptain(ICity city) {
		for (CaptainTownMapping mapped : captainMapping) {
			if (mapped.getTown().equals(city)) {
				return Optional.of(mapped.getCaptain());
			}
		}
		return Optional.empty();
	}

    /**
     * Retrieve the arrival date of a captain in a town. The captain must be present in a town,
     * otherwise an IllegalStateException is thrown.
     * @param captain captain residing in a town
     * @return time of his arrival.
     */
    public DateTime getArrivalDate(ICaptain captain) {
        for (CaptainTownMapping mapping : captainMapping) {
            if (mapping.getCaptain().equals(captain)) {
                return mapping.getArrivalDate();
            }
        }
        throw new IllegalStateException("The captain is not present in any town");
    }
	/**
	 * Remove the captain from the availables.
	 * @param captain that is hired.
	 * @param city where the captain was hired.
	 */
	public void hireCaptain(ICaptain captain, ICity city) {
		for (Iterator<CaptainTownMapping> iterator = captainMapping.iterator(); iterator.hasNext();) {
			CaptainTownMapping mapped = iterator.next();
			if (mapped.getTown().equals(city) && mapped.getCaptain().equals(captain)) {
				iterator.remove();
				break;
			}
		}
	}

	/**
	 * Retrieve a list of all captains that are not assigned to a town.
	 * @return
     */
	public List<ICaptain> getAllFreeCaptains() {
		 List<ICaptain> freeCaptains = new ArrayList<>();
		for (ICaptain captain : captains) {
			if (!captain.getAssignedShip().isPresent()) {
				freeCaptains.add(captain);
			}
		}
		return freeCaptains;
	}

	/**
	 * Find the current city of a captain.
	 * @param captain for which city is checked
	 * @return absent if the captain is in no town because he is hired on a ship.
     */
	public Optional<ICity> findCaptainsCurrentCity(ICaptain captain) {
		for (CaptainTownMapping mapping : captainMapping) {
			if (mapping.getCaptain().equals(captain)) {
				return Optional.of(mapping.getTown());
			}
		}
		return Optional.empty();
	}

}
