package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.annotation.SetType;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.event.NewPirateEvent;
import ch.sahits.game.openpatrician.model.people.INonFreeSeaPirate;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.*;
import java.util.stream.Collectors;

/**
 * State of all the sea pirates.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 1, 2013
 *
 */
@LazySingleton
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class SeaPiratesState {

	@Autowired
	@XStreamOmitField
	private Random rnd;
	@Autowired
	private IMap map;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;

	@Autowired
	private Date date;

	@SetType(ISeaPirate.class)
	private Set<ISeaPirate> pirates =  Sets.newHashSet();
	/**
	 * Add a pirate. The pirate may be a private pirate belonging only to one player.
	 * Post an update event to notify listiners of a new pirate.
	 * @param pirate
	 */
	public void add(ISeaPirate pirate) {
		pirates.add(pirate);
		clientServerEventBus.post(new NewPirateEvent(pirate));
	}

	/**
	 * Remove a pirate from the list.
	 * @param pirate
     */
	public void remove(ISeaPirate pirate) {
		pirates.remove(pirate);
	}

	public void replace(INonFreeSeaPirate pirate, ISeaPirate replaceWith) {
		pirates.remove(pirate);
		pirates.add(replaceWith);
	}
	/**
	 * Retrieve immutable list of non free pirates for the player
	 * @param player
	 * @return
	 */
	public List<ISeaPirate> getNonFreePirates(IPlayer player) {
		return Collections.unmodifiableList(pirates.stream()
				.filter(p -> p instanceof INonFreeSeaPirate && ((INonFreeSeaPirate)p).getHiringPlayer().equals(player))
				.collect(Collectors.toList()));
	}

	/**
	 * Retrieve all pirates.
	 * @return
     */
	public Set<ISeaPirate> getAllPirates() {
		return new HashSet<>(pirates);
	}
	/**
	 * Retrieve immutable list of free pirates.
	 * @return
	 */
	public Set<ISeaPirate> getFreePirates(){
		return pirates.stream().filter(p -> !(p instanceof INonFreeSeaPirate)).collect(Collectors.toSet());
	}
//	/**
//	 * Create an new pirate anywhere on the sea
//	 * @param name of the pirate
//	 * @param ship of the pirate
//	 * @param hiree player who hired the pirate
//	 * @param nearCity city near which the pirate should roam or null.
//	 * @return
//	 */
//	public INonFreeSeaPirate createNewPirate(String name, IShip ship, IPlayer hiree, ICity nearCity) {
//		NonFreeSeaPirate pirate = (NonFreeSeaPirate) context.getBean("nonFreeSeaPirate", new Object[]{hiree, date.getCurrentDate(), nearCity});
//		pirate.setName(name);
//		pirate.setShip(ship);
//		Point2D p;
//		if (nearCity != null) {
//			p = locationProvider.getRandomPointAtSeaNear(nearCity.getCoordinates());
//		} else {
//			p = locationProvider.getRandomPointAtSea();
//		}
//		pirate.getShip().setLocation(p);
//		return pirate;
//	}

	/**
	 * Find a successful free pirate
	 * @return
	 */
	public ISeaPirate findSuccessfulPirate() {
		Preconditions.checkArgument(!getFreePirates().isEmpty(), "There are no free pirates");
		int maxKill = 0;
		ISeaPirate successful = null;
		for (ISeaPirate pirate : getFreePirates()) {
			if (pirate.getNumberOfKills()>=maxKill) {
				successful = pirate;
				maxKill = pirate.getNumberOfKills();
			}
		}
		return successful;
	}
}
