package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import com.google.common.base.Preconditions;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.ReadOnlyIntegerProperty;

/**
 * Ecapsulate the price computation.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 25, 2013
 *
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@LazySingleton
public class ComputablePriceV2 {
	private final IPriceCalculationV2 calculation = new BezierPriceCalculation();

	public int calculateSellAmount(ITradable tradable, int availableAmount, int minPrice, int maxSellAmount) {
		int sellPriceAll = internalSellPriceCalculation(tradable, availableAmount, maxSellAmount);
        if (sellPriceAll > minPrice) {
            return maxSellAmount;
        }
        int halfAmount = (int) Math.round(maxSellAmount/2.0);
        int sellPriceHalfAmount = internalSellPriceCalculation(tradable, availableAmount, halfAmount);
        if (sellPriceHalfAmount == minPrice) {
            return halfAmount;
        }
        int bottom = 1;
		int sellPriceBottom = internalSellPriceCalculation(tradable, availableAmount, bottom);
		if (sellPriceBottom < minPrice || availableAmount == bottom) {
			return 0;
		}
        return calculateSellAmountInHalf(tradable, availableAmount, minPrice, maxSellAmount, sellPriceHalfAmount, halfAmount, bottom);
    }

    private int calculateSellAmountInHalf(ITradable tradable, int availableAmount, int minPrice, int maxSellAmount, int sellPriceHalfAmount, int halfAmount, int bottom) {
		if (sellPriceHalfAmount > minPrice) {
            // could sell more than half
            if (maxSellAmount == halfAmount + 1) {
                return halfAmount;
            } else {
                return calculateSellAmount(tradable, availableAmount, minPrice, halfAmount, maxSellAmount);
            }
        } else {
            // must sell less then half
            if (halfAmount == bottom + 1) {
                return bottom;
            } else {
                return calculateSellAmount(tradable, availableAmount, minPrice, 1, halfAmount);
            }
        }
    }

    private int calculateSellAmount(ITradable tradable, int availableAmount, int minPrice, int bottom, int top) {
        int halfAmount = (int) Math.round((top - bottom+1) / 2.0) + (bottom - 1);
        int sellPriceHalfAmount = internalSellPriceCalculation(tradable, availableAmount, halfAmount);
        if (sellPriceHalfAmount == minPrice) {
            return halfAmount;
        }
		if (bottom == halfAmount - 1) {
			return bottom;
		}
        int sellPriceAll = internalSellPriceCalculation(tradable, availableAmount, top);
        return calculateSellAmountInHalf(tradable, availableAmount, minPrice, top, sellPriceAll, halfAmount, bottom);
    }


    /**
	 * Internal calculation of the sell price.
	 * @param tradable good that is traded
	 * @param availableAmount amount of the good that is in store
	 * @param amountToSell amount to be sold
	 * @return sell price for one item of that good
	 */
	int internalSellPriceCalculation(ITradable tradable, int availableAmount, int amountToSell) {
		Preconditions.checkArgument(amountToSell > 0, "The amount to sell must be larger than 0 (was "+availableAmount+") for "+tradable.name()+", is "+amountToSell);
		Preconditions.checkArgument(availableAmount >= 0, "The available amount must be positive for "+tradable.name()+", is "+ availableAmount);
        int marketSaturation = tradable.getMarketSaturationForSelling();
        if (availableAmount>= marketSaturation){
			return tradable.getMinValueSell();
		}
		// Take amounts
		int firstItemPrice = (int)Math.rint(calculation.computePrice(tradable, false, availableAmount, 0,null,null));
		if (amountToSell>1){
			int lastItemPrice =  (int)Math.rint(calculation.computePrice(tradable, false, availableAmount+amountToSell, 0,null,null));
			return (int) Math.rint((firstItemPrice+lastItemPrice)/2.0);
		} else {
			return firstItemPrice;
		}
	}
	/**
	 * Internal method to calculate the buy price.
	 * @param tradable good that is to be traded
	 * @param availableAmount amount of the good that is in store
	 * @param amountToBuy amount to be bought
	 * @return buy price for one item of that good.
	 */
	int internalBuyPriceCalculation(ITradable tradable, int availableAmount, int amountToBuy) {
		Preconditions.checkArgument(amountToBuy > 0, "The amount to buy must be larger than 0 for "+tradable.name());
		Preconditions.checkArgument(availableAmount >= 0, "The available amount must be positive for "+tradable.name());
		if (availableAmount==0){
			return 0;
		}
		if (availableAmount>=tradable.getMarketSaturationForSelling()){
			return tradable.getMinValueBuy();
		}
		// Take amounts
		int firstItemPrice =  (int)Math.rint(calculation.computePrice(tradable, true, availableAmount,0,null,null));
		if (amountToBuy>1){
			// TODO make sure the value is not above the limit
			int lastItemPrice =  (int)Math.rint(calculation.computePrice(tradable, true, availableAmount-amountToBuy,0,null,null));
			return (int) Math.rint((firstItemPrice+lastItemPrice)/2.0);
		} else {
			return firstItemPrice;
		}
	}
	/**
	 * Integer binding calculation for the buy price.
	 * @param tradable good that is to be traded
	 * @param availableAmount amount of the good that is in store
	 * @param amountToBuy amount to be bought
	 * @return buy price for one item of that good.
	 */
	public int buyPrice(ITradable tradable, final ReadOnlyIntegerProperty availableAmount, final IntegerBinding amountToBuy) {
		return internalBuyPriceCalculation(tradable, availableAmount.get(), amountToBuy.get());
	}

	/**
	 * Calculate the amount between 0 and <code>maxAmount</code> that can be bought for at max <code>maxPrice</code>
	 * @param tradable that should be bought
	 * @param availableAmount amount that is available
	 * @param maxPrice maximum buy price
	 * @param maxAmount maximum amount to be bought
	 * @param  maxCash maximum amount of cash available.
     * @return maximal amount that can be bought
     */
	public int calculateBuyAmount(ITradable tradable, int availableAmount, int maxPrice, int maxAmount, long maxCash) {
		if (availableAmount <= 0 || maxCash <= 0) {
			return 0;
		}
		int firstItemPrice =  (int)Math.rint(calculation.computePrice(tradable, true, availableAmount,0,null,null));
		if (firstItemPrice == maxPrice) {
			if (firstItemPrice > maxCash) {
				return 0;
			} else {
				return 1;
			}
		}
		if (firstItemPrice > maxPrice) {
			return 0;
		}
		int top = Math.min(maxAmount, availableAmount);
		if (top <= 1) {
			return 0;
		}
		return calculateBuyAmount(tradable, availableAmount, maxPrice, 1, top, firstItemPrice, maxCash);

	}

	/**
	 * Calculate the maximum amount that is afordable by recustion. The price at the <code>bottom</code>
	 * end has already been checked and is affordable, so the affordable amount is larger or equal to that.
	 * @param tradable that should be bought
	 * @param availableAmount amount available
	 * @param maxPrice maximum price that is affordable
	 * @param bottom amount at the bottom of the range to be checked.
     * @param top amount at the top of the range that should be checked.
     * @param  maxCash maximum amount of cash available.
     * @return
     */
	private int calculateBuyAmount(ITradable tradable, int availableAmount, int maxPrice,  int bottom, int top, int firstItemPrice, long maxCash) {
		Preconditions.checkArgument(bottom < top,"The top value must be larger than the bottom value");
        int available = availableAmount - top + 1;
        int lastItemPrice =  (int)Math.rint(calculation.computePrice(tradable, true, available,0,null,null));
		int priceAtTop =  (int) Math.rint((firstItemPrice+lastItemPrice)/2.0);
		if (priceAtTop <= maxPrice && priceAtTop * top < maxCash) {
			return top;
		}
		if (top == bottom + 1) {
			return bottom;
		}
		int middleAmount = (int) Math.round((top - bottom+1) / 2.0) + (bottom - 1);
        available = availableAmount - middleAmount + 1;
        int priceMiddleItem = (int)Math.rint(calculation.computePrice(tradable, true, available,0,null,null));
		int priceAtMiddle =  (int) Math.rint((firstItemPrice+priceMiddleItem)/2.0);
		if (priceAtMiddle == maxPrice && priceAtMiddle < maxCash) {
			return middleAmount;
		}
		if (priceAtMiddle > maxPrice || priceAtMiddle * middleAmount > maxCash) {
			if (middleAmount == bottom + 1) {
				return bottom;
			} else {
				return calculateBuyAmount(tradable, availableAmount, maxPrice, bottom, middleAmount, firstItemPrice, maxCash);
			}
		} else {
			if (middleAmount == top - 1) {
				return middleAmount;
			} else {
				return calculateBuyAmount(tradable, availableAmount, maxPrice, middleAmount, top, firstItemPrice, maxCash);
			}
		}
	}

	/**
	 * Integer binding calculation of the sell price.
	 * @param tradable good that is traded
	 * @param availableAmount amount of the good that is in store
	 * @param amountToSell amount to be sold
	 * @return sell price for one item of that good
	 */
	public int sellPrice(ITradable tradable, final ReadOnlyIntegerProperty availableAmount, final IntegerBinding amountToSell) {
		return internalSellPriceCalculation(tradable, availableAmount.get(), amountToSell.get());
	}
}
