package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.model.IAmountable;

/**
 * Interface defining a ware. The ware provides general information on
 * the item it represents. It also provides methods to compute
 * the sell and buy price of the ware based on the available amount.
 * The price computation uses the following constants:<br>
 * <code>MAX</code>: the maximum price for the item if it is not available in the market.
 * Here the function has a glitch for the buy price: If no item is available
 * it cannot be sold and the price is therefore 0 (infinity would be logical correct
 * but is not very understandable in the game context).<br>
 * <code>MIN</code>: the minimal price of the item if more than <code>SATURATION</code> items
 * are available in the market. As more items are available in the market the price approaches
 * this value asymptoticly.<br>
 * <code>SATURATION</code>: amount of items when the the need is fullfilled. If there are more items
 * available in the market than needed all have the <code>MIN</code> price. This is a further
 * simplification because the price normally drops further if there are much more items available.
 * The curve would bee leveling out at <code>SATURATION</code> and then dropping further.<br>
 * <code>ABS_MIN</code>: the absolut minimum is the minimal sell price at <code>SATURATION</code>.<br>
 * <pre>
 *   (MAX-MIN)*(MIN+1)
 * --------------------- + ABS_MIN
 * e^(x*15/(SATURATION*2))+MIN
 * </pre>
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jan 15, 2011
 *
 */
public interface IWare extends IAmountable, ITradable {

	/**
	 * Retrieve the size of the ware converted into the smallest size unit (barrels)
	 * @return
	 */
	short getSizeAsBarrels();
	/**
	 * Check if the ware size is barrel or not.
	 * @return true if the ware is stored in barrels
	 */
	boolean isBarrelSizedWare();

	/**
	 * Retrieve the moderate max buy price of the ware.
	 * @return
     */
	int getMaxBuyPriceModerate();

	/**
	 * Retrieve the offensive max buy price of the ware.
	 * @return
     */
	int getMaxBuyPriceOffensive();

	/**
	 * Retrieve the rabiat max buy price for the ware.
	 * @return
     */
	int getMaxBuyPriceRabiat();

	/**
	 * Retrieve the amount in percentage the AI should buy of a certain ware when
	 * compared to the ship size.
	 * @return
     */
	double getAiMaxBuyPercentage();

	/**
	 * Retrieve the maximum absolute amount of the ware that should be loaded on a ship.
	 * @return
     */
	int getAiMaxBuyAbsolute();

}