package ch.sahits.game.openpatrician.model.sea;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import javafx.geometry.Point2D;
import javafx.scene.shape.Path;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * This iterable collection stores all vessels that are currently travelling.
 * Vessels should be added when they start their travel and be removed when they reach their destination or
 * otherwise stop traveling (e.g. are sunk).
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 08, 2016
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
@Component
@Lazy
public class TravellingVessels implements ITravellingVessels {
    @MapType(key = INavigableVessel.class, value = TravellingVessel.class)
    private final Map<INavigableVessel, TravellingVessel> vessels = new ConcurrentHashMap<>();

    /**
     * The <code>vessel</code> starts it's travel and must be added to this collection together with its data.
     * @param vessel that starts the travel
     * @param path the Bezière path representing the route.
     * @param points the list of points that make up the path along which the <code>vessel</code> is travelling.
     */
    public void addVessel(INavigableVessel vessel, Optional<Path> path, List<Point2D> points) {
        TravellingVessel tv = new TravellingVessel(vessel);
        tv.setCalculatablePath(points);
        if (path.isPresent()) {
            tv.setDrwawablePath(path.get());
        }
        vessels.put(vessel, tv);
    }

    /**
     * The vessel is no longer travelling and should no longer be part of this collection.
     * @param vessel
     */
    public void remove(INavigableVessel vessel) {
        vessels.remove(vessel);
    }

    @Override
    public Iterator<INavigableVessel> iterator() {
        return vessels.keySet().iterator();
    }

    /**
     * Retrieve the meta date for a travelling vessel.
     * @param vessel
     * @return
     */
    @Override
    public TravellingVessel getTravellingVessel(INavigableVessel vessel) {
        return vessels.get(vessel);
    }

    /**
     * Check if the vessel is currently travelling.
     * @param vessel
     * @return
     */
    @Override
    public boolean isTravelling(INavigableVessel vessel) {
        return vessels.containsKey(vessel);
    }
}
