package ch.sahits.game.openpatrician.model.ship.impl;

import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.collections.NonReplacableMap;
import ch.sahits.game.openpatrician.javafx.bindings.LateIntegerBinding;
import ch.sahits.game.openpatrician.model.AmountableProvider;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.EShipSide;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IShipWeaponsLocation;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.openpatrician.util.service.LUIDProvider;
import ch.sahits.game.openpatrician.util.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.util.spring.DependentValue;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.geometry.Point2D;
import lombok.Getter;
import lombok.Setter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;

import static com.google.common.collect.Maps.newHashMap;
@Prototype
public abstract class Ship implements IShip {
	@XStreamOmitField
	private final Logger logger = LogManager.getLogger(getClass());
	/** Amount of barrels space a sailor needs */
	@DependentValue("sailor.space.on.ship")
	private double spacePerSailor = 0.5;
	@DependentValue("passenger.space.on.ship")
	private int passengerSpace = 2;
	@Getter
	private int distanceInKmForOneHealthPointReduction;
	@Autowired
	private LUIDProvider luidProvider;

	/** State of the design standard level */
    @Getter
	protected EShipUpgrade shipUpgradeLevel;
	/** Name of the ship */
    @Getter
    @Setter
	protected String name;
	/** Store the wares loaded on the ship together with their amount. The amount is ware specific*/
	@MapType(key=IWare.class, value = AmountablePrice.class)
	private Map<IWare,AmountablePrice<IWare>> loadedWare = new NonReplacableMap<>();
	/**
	 * Binding representing the current load.
	 */
    @XStreamOmitField
	private LateIntegerBinding loadBinding = createLoadBinding();

	private final double topSpeed;

	@Autowired
	@XStreamOmitField
	private AmountableProvider amountableProvider;
	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;
    @XStreamOmitField
	private BooleanBinding passengerPresent = new BooleanBinding() {
		@Override
		protected boolean computeValue() {
			return passenger.isPresent();
		}
	};



	@MapType(key = IWeapon.class, value = Integer.class)
	private Map<IWeapon,Integer> loadedWeapons = newHashMap();
    @Getter
    @Setter
	private IShipWeaponsLocation shipWeaponsLocation;
    /** Initial value of the ship */
    private final int initialValue;
	@Getter
    private final int minNumberOfSailors;
    private final int maxSailors;
    @Getter
	private Optional<ICaptain> captian = Optional.empty();
    @Getter
    private Optional<IPerson> passenger = Optional.empty();
    /**
	 * Fitness of the ship. This is reduced as it is damaged
	 */
	protected int fitness = 100;

	private IntegerProperty nbSailors = new SimpleIntegerProperty(0);
    @Setter
    @Getter
	private Point2D location = null;
	private IShipOwner owner = null;
	@Setter
	@Getter
	private boolean available = true;
	/** Space occupied by wares not owned by the owner */
	@Getter
	@Setter
	private int occupiedSpace;
	/** Size in barrels */
	@Getter
	private final int size;
	private int reductionLevel1;
	private int reductionLevel2;
	@Getter
	private String uuid;
	private boolean pirateFlag;

	protected Ship(Properties shipConfiguration) {
		this.initialValue = Integer.parseInt(shipConfiguration.getProperty("initialValue"));
		this.minNumberOfSailors = Integer.parseInt(shipConfiguration.getProperty("minNumberOfSailors"));
		this.maxSailors = Integer.parseInt(shipConfiguration.getProperty("maxSailors"));
		this.size = Integer.parseInt(shipConfiguration.getProperty("size"));
		this.reductionLevel1 = Integer.parseInt(shipConfiguration.getProperty("reductionLevel1"));
		this.reductionLevel2 = Integer.parseInt(shipConfiguration.getProperty("reductionLevel2"));
		this.topSpeed = Double.parseDouble(shipConfiguration.getProperty("speedkmperh"));
		this.distanceInKmForOneHealthPointReduction = Integer.parseInt(shipConfiguration.getProperty("distanceInKmForOneHealthPointReduction"));
	}

	@PostConstruct
	private void init() {
		if (uuid == null) { // avoid overriding when the UUID is loaded
			uuid = luidProvider.getNextLUID();
		}
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			logger.warn("Failed to initialize DependentValue annotated fields");
		}
	}


	@Override
	public int getMaxNumberOfSailors() {
		int availableSpace = getCapacity();
		int spaceForSailors = (int)Math.rint(availableSpace/ spacePerSailor);
		return Math.min(maxSailors, spaceForSailors);
	}


	@Override
	public void setCaptain(ICaptain captain) {
		this.captian = Optional.ofNullable(captain);
	}

	@Override
	public boolean isUpgradable() {
		return shipUpgradeLevel != EShipUpgrade.LEVEL2;
	}

	@Override
	public void upgrade() {
		EShipUpgrade[] levels = EShipUpgrade.values();
		for (int i = 0; i < levels.length; i++) {
			if (levels[i]==shipUpgradeLevel){
                shipUpgradeLevel = levels[i+1];
				break; // only one update allowed
			}
		}
	}


	@Override
	public Set<IWare> getLoadedWares() {
		return loadedWare.keySet();
	}

	/**
	 * {@inheritDoc}
	 * This method is not thread safe as it is only intended to be accessed by one thread at a time
	 */
	@Override
	public int load(IWare ware, int amount, int avgPrice) {
		final short sizeInBarrels = ware.getSizeAsBarrels();
		amount = Math.abs(amount);
		int cap = ensureCapacity(amount*sizeInBarrels);
		// Make sure that for wares in other sizes than barrels we only load
		// complete loads.
		if (cap!=amount && sizeInBarrels!=1){
			cap = (cap/sizeInBarrels)*sizeInBarrels;
		}
		AmountablePrice<IWare> available = getWare(ware);
		available.add(cap/ware.getSizeAsBarrels(), avgPrice);
		return cap/sizeInBarrels;
	}

	/**
	 * Check if the amount can be loaded
	 * @param amount
	 * @return
	 */
	private int ensureCapacity(int amount) {
		return Math.max(Math.min(amount, getCapacity()), 0);
	}

	/**
	 * {@inheritDoc}
	 * This method is not thread safe as it is only intended to be accessed by one thread at a time
	 */
	@Override
	public int unload(IWare ware, int amount) {
		if (!loadedWare.containsKey(ware)){
			return 0; // nothing to unload
		}
		amount = Math.abs(amount);
		// convert to barrels
		amount = amount*ware.getSizeAsBarrels();
		final int loaded = loadedWare.get(ware).getAmount()*ware.getSizeAsBarrels();
		int unloaded = Math.min(loaded, amount);
		if (unloaded==loaded){ // unloaded completely
			AmountablePrice<IWare> available = loadedWare.get(ware);
			available.reset();
		} else {
			AmountablePrice<IWare> available = loadedWare.get(ware);
			available.remove(unloaded/ware.getSizeAsBarrels());
		}
		return unloaded/ware.getSizeAsBarrels();
	}


	private int internalLoadCalculation() {
		int sum=0;
		for (Entry<IWare,AmountablePrice<IWare>> entry : loadedWare.entrySet()) {
			int amount = entry.getValue().getAmount();
			int barrelSize = entry.getKey().getSizeAsBarrels();
			sum += amount*barrelSize;
		}
		return sum;
	}
	@Override
	public IntegerBinding getLoadBinding() {
		return loadBinding;
	}

	/**
	 * Clear all loaded wares. This method is only intended for testing
	 */
	protected void clearLoadedWares() {
		for (AmountablePrice<IWare> amountable : loadedWare.values()) {
			amountable.reset();
		}
	}

	@Override
	public AmountablePrice<IWare> getWare(IWare ware) {
		if (!loadedWare.containsKey(ware)){
			AmountablePrice<IWare> amountable = amountableProvider.createWareAmountable();
			loadedWare.put(ware, amountable);
			loadBinding.bind(amountable.amountProperty());
		}
		return loadedWare.get(ware);
	}
	/**
	 * Retrieve the side of the next free slot where the weapon can be placed.
	 * check first port and then starboard. There is no free solt on either side,
	 * null will be returned.
	 * @param weapon to be placed
	 * @return {@link EShipSide#PORT}, {@link EShipSide#STARBOARD} or null
	 */
	@SuppressWarnings("unused")
	private EShipSide getNextFreeSide(IWeapon weapon){
		// TODO implement
		return null;
	}
	/**
	 * Retrieve the next free slot for the weapon on the side. The slots are checked
	 * from stern to bow. If there is no free slot a negative number will be returned
	 * @param weapon to be placed
	 * @param side to be checked {@link EShipSide#PORT} or {@link EShipSide#STARBOARD}
	 * @return slot index or negative number
	 */
	@SuppressWarnings("unused")
	private int getNextFreeSlot(IWeapon weapon, EShipSide side){
		// TODO implement
		return -1;
	}

	@Override
	public boolean hasWeapons() {
		return false; // TODO aho Jun 1, 2013: solve this through the ShipWeaponsLocation
	}
	@Override
	public int getValue() {
		return (int)Math.rint(initialValue*fitness/100.0);
	}


	@Override
	public int getNumberOfSailors() {
		return nbSailors.get();
	}


	@Override
	public void setNumberOfSailors(int nbSailors) {
		this.nbSailors.set(nbSailors);
	}

    @Override
    public IntegerProperty numberOfSailorsProperty() {
        return nbSailors;
    }

    @Override
	public void move(IWeapon weaponType, int amount){
		if (!loadedWeapons.containsKey(weaponType)) {
			loadedWeapons.put(weaponType, 0);
		}
		if (amount<0) {
			// remove
			boolean check = -amount<=loadedWeapons.get(weaponType);
			Preconditions.checkArgument(check, "Cannot remove more than available");
		}
		int amountable = loadedWeapons.get(weaponType);
		loadedWeapons.put(weaponType, amountable+amount);
	}

	@Override
	public int getWeaponAmount(IWeapon weaponType) {
		if (loadedWeapons.containsKey(weaponType)) {
			return loadedWeapons.get(weaponType);
		} else {
			return 0;
		}
	}
	/**
	 * Move a passenger to the ship. The ship may only contain one passanger.
	 * @param passanger
	 */
	@Override
	public void addPassenger(IPerson passanger) {
		Preconditions.checkArgument(!passenger.isPresent(), "Passenger already present");
		passenger = Optional.of(passanger);
		passengerPresent.invalidate();
	}
	@Override
	public IShipOwner getOwner() {
			return owner;
	}

	@Override
	public void setOwner(IShipOwner owner) {
		Preconditions.checkNotNull(owner, "Owner of a ship may not be null");
		this.owner = owner;
	}

	/**
	 * Passenger leave ship.
	 */
	@Override
	public void leavePassanger(){
		passenger = Optional.empty();
		passengerPresent.invalidate();
	}

	@Override
	public int getCapacity() {
		int passengerSpace = passenger.isPresent() ? this.passengerSpace : 0;
		return size-getUpgradeSpaceReduction()-internalLoadCalculation()-getOccupiedSpace() - passengerSpace;
	}

	@Override
	public int getUpgradeSpaceReduction(){
		switch (shipUpgradeLevel) {
			case NONE:
				return 0;
			case LEVEL1:
				return reductionLevel1;
			case LEVEL2:
				return reductionLevel2;
			default:
				return 0;
		}
	}

	private LateIntegerBinding createLoadBinding() {
		return new LateIntegerBinding() {

			@Override
			protected int computeValue() {
				return internalLoadCalculation();
			}
		};
	}

    public BooleanBinding passengerPresentProperty() {
        return passengerPresent;
    }
	@Override
	public int getDamage() {
		return 100 - fitness;
	}

	@Override
	public void damage(int damage) {
        Preconditions.checkArgument(damage > 0, "The damage must be positive");
		fitness = Math.max(0, fitness - damage);
	}

    @Override
    public int getLoadableSpace() {
        return getSize() - getUpgradeSpaceReduction();
    }

    @Override
    public double getTopSpeed() {
        return topSpeed;
    }

	@Override
	public boolean getPirateFlag() {
		return pirateFlag;
	}

	@Override
	public void togglePirateFlag() {
		pirateFlag = !pirateFlag;
	}

	@Override
	public double getCurrentSpeed() {
		if (getDamage() == 0) {
			return topSpeed;
		}
		return Math.max(0.5, topSpeed / getDamage()/ 100.0);
	}
}