package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.time.Period;
import java.time.temporal.ChronoUnit;

/**
 * Service concerning the date.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on May 15, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class DateService {

    @Autowired
    private Date date;

    /**
     * Check if two dates represent the same day
     * @param date1
     * @param date2
     * @return
     */
    public boolean isSameDay(LocalDateTime date1, LocalDateTime date2) {
        
        return date1.getDayOfYear() == date2.getDayOfYear();
    }
    /**
     * Check if the date is today.
     * @param date
     * @return
     */
    public boolean isToday(LocalDateTime date) {
        return date.getDayOfYear() == this.date.getCurrentDate().getDayOfYear();
    }

    /**
     * Check if a date lies in the past.
     * @param date to be checked.
     * @return true if the date is before the current game date
     */
    public boolean isPast(LocalDateTime date) {
        return date.isBefore(this.date.getCurrentDate());
    }

    /**
     * Calculate the age.
     * @param birthDate
     * @return
     */
    public int getAge(LocalDateTime birthDate) {
        return Period.between(birthDate.toLocalDate(), date.getCurrentDate().toLocalDate()).getYears();
    }

    /**
     * Calculate the percentage of days passed between the start and end date.
     * @param start date
     * @param end date
     * @return percentage of days passed in [0,1]
     */
    public double getPercentageBetweenDates(LocalDateTime start, LocalDateTime end) {
        Preconditions.checkArgument(end.isAfter(start), "End date must be after start date");
        double duration =  ChronoUnit.DAYS.between(start, end);
        double pastDays =  ChronoUnit.DAYS.between(start, date.getCurrentDate());
        return pastDays/duration;
    }

}
