package ch.sahits.game.openpatrician.model.city;

import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IShip;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

/**
 * Interface defining the model of the ship yard.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Mar 9, 2013
 *
 */
public interface IShipyard {
    /**
     * Retrieve the city the shipyard belongs to.
     * @return
     */
    ICity getCity();
	/**
	 * Retrieve an immutable list of the ships that are ordered
	 * to be built on this shipyard.
	 * @return immutable list of ship due dates.
	 */
	List<IShipDueDate> getShipBuildingList();
	/**
	 * Retrieve the date when the building of a new ship will be finished.
	 * @param type of the ship to be built
	 * @return end date of the building
	 */
	LocalDateTime getBuildCompleteDate(EShipType type);

    /**
     * Add an order for building a new ship to the list.
     * @param dueDate when the ship is finished building
     * @param task that is executed on completion.
     */
    void addShipBuildingOrder(IShipDueDate dueDate, Optional<TimedTask> task);

    /**
     * Add an order for repairing a ship to the list.
     * @param dueDate when the ship is finished repairing
     * @param task that is executed on completion.
     */
    void addShipRepairOrder(IShipDueDate dueDate, Optional<TimedTask> task);
    /**
     * Add an order for refitting a ship to the list.
     * @param dueDate when the ship is refitted
     * @param task that is executed on completion.
     */
    void addShipRefitOrder(IShipDueDate dueDate, Optional<TimedTask> task);
	/**
	 * Cancel the order of the building, finished on the given date.
	 * @param finish date
	 */
	void cancelShipBuildingOrder(LocalDateTime finish);
	/**
	 * Retrieve the list of all ships that are to be repaired.
	 * @return immutable list of all the repairing ships
	 */
	List<IShipDueDate> getShipRepairList();
	/**
	 * Retrieve the list of all ships that are to be upgraded.
	 * @return immutable list of all the upgrading ships
	 */
	List<IShipDueDate> getShipUpgradeList();
	/**
	 * Cancel the repair of the ship. No damage will be repaired
	 * @param ship which's repair order is to be canceled
	 */
	void cancelRepair(IShip ship);

	/**
	 * Calculate the repair costs.
	 * @param repairTime
	 * @param materialCosts
	 * @return
	 */
	int calculateRepairCosts(int repairTime, int materialCosts);
	/**
	 * Calculate the material costs for the repair.
	 * @param ship
	 * @param damage
	 * @return
	 */
	int calculateRepairMaterialCosts(IShip ship, int damage);
	/**
	 * Calculate the labor costs for the refit based on the ship type and how many levels
	 * should be upgraded.
	 * @param type
	 * @param levels
	 * @return
	 */
	int calculateRefitCosts(EShipType type, int levels);

    /**
     * Calculate the time in days it takes for refitting.
     * @param type destination level
     * @return time in days.
     */
    int calculateRefitTime(EShipType type);
	/**
	 * Retrieve a list of all the types of ships this yard can build.
	 * @return
	 */
	EShipType[] getBuildableShipTypes();
	/**
	 * Calculate the labor costs for building a new ship of the indicated type.
	 * @param type of the ship
	 * @return costs.
	 */
	int calculateConstructionCosts(EShipType type);
	/**
	 * Calculate the build time for the ship of type.
	 * @param type
	 * @return
	 */
	int calculateBuildTime(EShipType type);
	/**
	 * Calculate the repair time for the ship of type.
	 * @param type
	 * @return
	 */
	int calculateRepairTime(EShipType type);

	/**
	 * Remove the repair entry once it is completed.
	 * @param ship that is to be repaired.
     */
	void removeCompletedRepair(IShip ship);
	/**
	 * Remove the build entry once it is completed.
	 * @param ship that is to be built.
	 */
	void removeCompletedConstruction(IShip ship);
	/**
	 * Remove the upgrade entry once it is completed.
	 * @param ship that is to be upgraded.
	 */
	void removeCompletedUpgrade(IShip ship);
}
