package ch.sahits.game.openpatrician.model.city;

import ch.sahits.game.openpatrician.model.product.AmountablePriceDouble;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.model.product.IWare;

import java.util.HashMap;
import java.util.Map;

/**
 * This is the local storage for the production workshops.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on May 14, 2016
 */
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.SERIALIZABLE_BEAN})
@Prototype
public class ProductionStorage {
    @MapType(key = IWare.class, value = Double.class)
    private Map<IWare, AmountablePriceDouble> storage = new HashMap<>();

    /**
     * Retrieve the amount that is stored of a specific ware.
     * @param ware
     * @return
     */
    public double getStored(IWare ware) {
        if (storage.containsKey(ware)) {
            return storage.get(ware).getAmount();
        } else {
            return 0;
        }
    }

    /**
     * Retrieve the average price of the stored amount.
     * @param ware for which the price is to be looked up
     * @return average price or 0, if the ware is not available.
     */
    public double getAvgPrice(IWare ware) {
        if (storage.containsKey(ware)) {
            return storage.get(ware).getAVGPrice();
        } else {
            return 0;
        }
    }

    /**
     * Transfer wares to/from the storage. Negative <code>amount</code>
     * values mean withdrawing from the storage.
     * @param ware that is to be transfered
     * @param amount to be moved
     * @param avgPrice of the ware
     * @return the amount that was actually transfered
     */
    public double transfer(IWare ware, double amount, double avgPrice) {

        AmountablePriceDouble amountablePrice;
        double stored = 0.0;
        if (storage.containsKey(ware)) {
            amountablePrice = storage.get(ware);
            stored = amountablePrice.getAmount();
        } else {
            amountablePrice = new AmountablePriceDouble();
            storage.put(ware, amountablePrice);
        }
        if (amount < 0) {
            amountablePrice.remove(-amount);
        } else {
            amountablePrice.add(amount, avgPrice);
        }
        double storedAfter = amountablePrice.getAmount();
        return storedAfter - stored;
    }
}
