package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.IgnoreOnDeserialisation;
import ch.sahits.game.openpatrician.data.xmlmodel.Production;
import ch.sahits.game.openpatrician.model.javafx.bindings.LateIntegerBinding;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.city.EKontorType;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.FoundNewSettlement;
import ch.sahits.game.openpatrician.model.impl.WareHolding;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.utilities.service.PropertyLoader;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import ch.sahits.util.ClassChecker;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Interner;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.geometry.Point2D;
import lombok.Getter;
import lombok.Setter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Random;
import java.util.ResourceBundle;

/**
 * Implementation of the city model. The model of the city should only be instanciated once.
 * A city is unique. therefore equality can be tested by identity.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jan 18, 2011
 *
 */
@Component
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class City extends WareHolding implements ICity {
	@XStreamOmitField
	private static final Logger LOGGER = LogManager.getLogger(City.class);
	@Autowired
	@XStreamOmitField
	private Random rnd;
	@Getter
	private final IWare[] effectiveProduction;
	@Getter
	private final IWare[] ineffectiveProduction;
	@Getter
    @ListType(IWare.class)
	private final List<IWare> imported;
	@Getter
	private String name;
	@Getter
	private final String nameTemplate;
	@Getter
	private final EKontorType kontorType;

	@Autowired
	@XStreamOmitField
	private Locale locale;
	@Autowired
	@XStreamOmitField
	private MessageSource messageSource;
	@Getter
	private Point2D coordinates;
	@MapType(key = IWare.class, value = LocalDateTime.class)
	private final Map<IWare, LocalDateTime> missingWares = new HashMap<>();
	/**
	 * Store the buildings in the city
	 */
	@ListType(IBuilding.class)
	private List<IBuilding> buildings = new ArrayList<>();
	/**
	 * Map holding the reputation of the different players identified by their UUID.
	 */
	@MapType(key = String.class, value = IReputation.class)
	private HashMap<String, IReputation> reputation = new HashMap<>();
	/**
	 * Store the contibutions of the players
	 */
	@MapType(key = String.class, value = Contributions.class)
	private Map<String, Contributions> playersContributions = new HashMap<>();
	/**
	 * Holding the population split by population classes
	 */
	@MapType(key = EPopulationClass.class, value = IntegerProperty.class)
	private final Map<EPopulationClass, IntegerProperty> population = new HashMap<>();
	@Getter
	@ListType(ICitizen.class)
	private final List<ICitizen> citizen = new ArrayList<>();
	@XStreamOmitField
	private LateIntegerBinding totalPopulation = null;

	private int roadTiles;

	@Getter
	@Setter
	private CityState cityState = null;
	@MapType(key = String.class, value = Boolean.class)
	private Map<String, Boolean> buildingPermission = new HashMap<>();
	@Autowired
	private Date date;
	@Autowired
	@XStreamOmitField
	protected Interner<Point2D> pointInterner;
	@Getter
	private final boolean riverCity;
	@Getter
	private final Point2D cityLabelOffset;
	@Getter
	private final ECityLabelAlignment cityLabelAlignment;

	public City(ch.sahits.game.openpatrician.data.xmlmodel.City city) {
		Production production = city.getProduction();
		Preconditions.checkNotNull(city.getName(), "The city does not define a name");
		nameTemplate = city.getName();
		Preconditions.checkArgument(!production.getEffective().getWare().isEmpty(), "The city " + nameTemplate + " does not contain the effectiveProduction property");
		Preconditions.checkArgument(!production.getIneffective().getWare().isEmpty(), "The city " + nameTemplate + " does not contain the ineffectiveProduction property");
		Preconditions.checkNotNull(city.getKontorType(), "The city " + nameTemplate + " does not define the kontor type.");
		Preconditions.checkNotNull(city.getLocation().getX(), "The city " + nameTemplate + " does not contain the city location");
		Preconditions.checkNotNull(city.getLocation().getY(), "The city " + nameTemplate + " does not contain the city location");
		int x = city.getLocation().getX();
		int y = city.getLocation().getY();
		imported = new ArrayList<>();
		effectiveProduction = new EWare[production.getEffective().getWare().size()];
		for (int i = 0; i < production.getEffective().getWare().size(); i++) {
			EWare ware = EWare.valueOf(production.getEffective().getWare().get(i).getValue());
			effectiveProduction[i] = ware;
			boolean imported = production.getEffective().getWare().get(i).isImported();
			if (imported) {
				this.imported.add(ware);
			}
		}
		ineffectiveProduction = new EWare[production.getIneffective().getWare().size()];
		for (int i = 0; i < production.getIneffective().getWare().size(); i++) {
			EWare ware = EWare.valueOf(production.getIneffective().getWare().get(i).getValue());
			ineffectiveProduction[i] = ware;
			boolean imported = production.getIneffective().getWare().get(i).isImported();
			if (imported) {
				this.imported.add(ware);
			}
		}
		coordinates = new Point2D(x, y);
		kontorType = EKontorType.valueOf(city.getKontorType());
		riverCity = city.isRiver();
		roadTiles = internalPopulationCount() / 10;
		cityLabelOffset = new Point2D(city.getOffset().getX(), city.getOffset().getY());
		if (city.getOffset().getAlignment() == null) {
			cityLabelAlignment = ECityLabelAlignment.LEFT;
		} else {
			cityLabelAlignment = ECityLabelAlignment.valueOf(city.getOffset().getAlignment().value());
		}
	}

	public City(FoundNewSettlement futureTown) {
		nameTemplate = futureTown.getName();
		int x = (int) Math.rint(futureTown.getLocation().getX());
		int y = (int) Math.rint(futureTown.getLocation().getY());
		coordinates = new Point2D(x, y);
		kontorType = EKontorType.FOUNDED_SETTLEMENT;
		effectiveProduction = new EWare[2];
		List<EWare> addedWares = new ArrayList<>();
		int nbWares = EWare.values().length;
		for (int i = 0; i < 2; i++) {
			EWare ware = EWare.values()[rnd.nextInt(nbWares)];
			while (addedWares.contains(ware)) {
				ware = EWare.values()[rnd.nextInt(nbWares)];
			}
			effectiveProduction[i] = ware;
			addedWares.add(ware);
		}
		ineffectiveProduction = new EWare[3];
		for (int i = 0; i < 3; i++) {
			EWare ware = EWare.values()[rnd.nextInt(nbWares)];
			while (addedWares.contains(ware)) {
				ware = EWare.values()[rnd.nextInt(nbWares)];
			}
			ineffectiveProduction[i] = ware;
			addedWares.add(ware);
		}
		riverCity = futureTown.isRiver();
		cityLabelOffset = futureTown.getCityLabelOffset();
		cityLabelAlignment = futureTown.getCityLabelAlignment();
		imported = new ArrayList<>();
	}


	@PostConstruct
	@IgnoreOnDeserialisation
	private void init() {
		this.name = messageSource.getMessage(nameTemplate, new Object[0], locale.getCurrentLocal()); // FIXME: 5/1/16 the localisation should not happen here as the model might be on the server and the client may have a different locale
		if (!containsWare(EWare.GRAIN)) {
			initWares();
		}
		if (population.isEmpty()) {
			initPopulation();
		}
		coordinates = pointInterner.intern(coordinates);
	}

	/**
	 * This Constructor is used by tests
	 */
	City(String configFileName) throws IOException {
		Properties props = PropertyLoader.loadProperties(configFileName);
		if (props.getProperty("effectiveProduction") == null) {
			throw new IOException("The property file " + configFileName + " does not contain the effectiveProduction property");
		}
		if (props.getProperty("ineffectiveProduction") == null) {
			throw new IOException("The property file " + configFileName + " does not contain the ineffectiveProduction property");
		}
		if (props.getProperty("name") == null) {
			throw new IOException("The property file " + configFileName + " does not contain the name property");
		}
		if (props.getProperty("kontorType") == null) {
			throw new IOException("The property file " + configFileName + " does not contain the kontorType property");
		}
		if (props.getProperty("location.x") == null || props.getProperty("location.y") == null) {
			throw new IOException("The property file " + configFileName + " does not contain the city location");
		}
		if (props.getProperty("effectiveProduction").trim().length() > 0) {
			String[] wareNames = props.getProperty("effectiveProduction").split(",");
			effectiveProduction = new EWare[wareNames.length];
			for (int i = 0; i < wareNames.length; i++) {
				effectiveProduction[i] = EWare.valueOf(wareNames[i]);
			}
		} else {
			effectiveProduction = null;
		}
		if (props.getProperty("ineffectiveProduction").trim().length() > 0) {
			String[] wareNames = props.getProperty("ineffectiveProduction").split(",");
			ineffectiveProduction = new EWare[wareNames.length];
			for (int i = 0; i < wareNames.length; i++) {
				ineffectiveProduction[i] = EWare.valueOf(wareNames[i]);
			}
		} else {
			ineffectiveProduction = null;
		}
		String n = props.getProperty("name");
		nameTemplate = n;
		ResourceBundle messages = ResourceBundle.getBundle("ModelMessages", locale.getCurrentLocal());  // FIXME: 5/1/16 the localisation should not happen here as the model might be on the server and the client may have a different locale
		name = messages.getString(n);
		kontorType = EKontorType.valueOf(props.getProperty("kontorType"));
		int x = Integer.parseInt(props.getProperty("location.x"));
		int y = Integer.parseInt(props.getProperty("location.y"));
		coordinates = new Point2D(x, y);
		date = new Date(2013);
		riverCity = false;
		cityLabelAlignment = ECityLabelAlignment.LEFT;
		cityLabelOffset = coordinates.add(new Point2D(-30, 20));
		imported = new ArrayList<>();
		initPopulation();
		initWares();
	}

	/**
	 * Init the amount of wares available in the city
	 * This method is protected so it can be overriden by subclasses for testing
	 */
	protected void initWares() {
		List<IWare> producing = new ArrayList<>(Arrays.asList(getEffectiveProduction()));
		producing.addAll(Arrays.asList(getIneffectiveProduction()));
		for (EWare ware : EWare.values()) {
			boolean isProducing = producing.contains(ware);
			boolean hasWare = isProducing || rnd.nextInt(7) % 7 != 0;
			if (hasWare) {
				int limit = ware.getMarketSaturationForSelling() / 2;
				if (isProducing) {
					limit = limit * 3;
				}
				int amount = rnd.nextInt(limit) + 1; // 1..limit
				addNewWare(ware, amount);
			} else {
				addNewWare(ware, 0);
			}
		}

	}

	/**
	 * Initialize the population of the different classes based on the properties
	 */
	private void initPopulation() {
		// TODO set the polulation structure based on the settings from the properties and the start year
		int pop = 1000 + (int) Math.abs(rnd.nextDouble() * 4000); // value between 1000 and 5000
		int diffPoor = (int) ((rnd.nextDouble() - 0.5) * 7);
		int poor = (int) ((60.0 + diffPoor) / 100 * pop); // about 60% poor
		int begger = calculateInitialBeggarCount(pop);
		setPopulation(begger, EPopulationClass.BEGGAR);
		setPopulation(poor, EPopulationClass.POOR);
		int medium = (pop - poor) * 2 / 3;
		int rich = pop - poor - medium;
		setPopulation(medium, EPopulationClass.MEDIUM);
		setPopulation(rich, EPopulationClass.RICH);

	}

	private int calculateInitialBeggarCount(int totalPopulation) {
		int beerWareAmounts = getWare(EWare.BEER).getAmount();
		int fishWareAmounts = getWare(EWare.FISH).getAmount();
		int grainWareAmounts = getWare(EWare.GRAIN).getAmount();

		int baseAmount = totalPopulation / 100;
		double factorAvailablility = 1;
		if (beerWareAmounts == 0) {
			factorAvailablility -= 0.25;
		}
		if (fishWareAmounts == 0) {
			factorAvailablility -= 0.25;
		}
		if (grainWareAmounts == 0) {
			factorAvailablility -= 0.25;
		}
		double amountFactor = 1;
		if (beerWareAmounts > 100) {
			factorAvailablility += 0.25;
		}
		if (fishWareAmounts > 50) {
			factorAvailablility += 0.25;
		}
		if (grainWareAmounts > 20) {
			factorAvailablility += 0.25;
		}
		double randomFactor = rnd.nextDouble() * 2 + 1; // [1,3]
		return (int) (baseAmount * factorAvailablility * amountFactor * randomFactor);

	}

	/**
	 * Count the population.
	 *
	 * @return
	 */
	private int internalPopulationCount() {
		int count = 0;
		for (IntegerProperty i : population.values()) {
			count += i.get();
		}
		return count;
	}

	@Override
	public IntegerBinding getPopulationBinding() {
		if (totalPopulation == null) {
			totalPopulation = new LateIntegerBinding() {

				@Override
				protected int computeValue() {
					return internalPopulationCount();
				}
			};
		}
		return totalPopulation;
	}

	/**
	 * Set the population count for a apopulation class
	 *
	 * @param population count
	 * @param popClass   population class
	 */
	@Override
	public void setPopulation(int population, EPopulationClass popClass) {
		SimpleIntegerProperty populationProperty = new SimpleIntegerProperty(population);
		this.population.put(popClass, populationProperty);
		((LateIntegerBinding) getPopulationBinding()).bind(populationProperty);
	}

	/**
	 * Retrieve the population count for a class
	 *
	 * @param popclass population class
	 * @return
	 */
	@Override
	public int getPopulation(EPopulationClass popclass) {
		return population.get(popclass).get();
	}

	@Override
	public IntegerProperty getPopulationProperty(EPopulationClass popclass) {
		return population.get(popclass);
	}


	@Override
	public List<IBuilding> getBuildings() {
		return Collections.unmodifiableList(buildings);
	}

	/**
	 * Add a new building to the city
	 *
	 * @param building
	 */
	@Override
	public void build(IBuilding building) {
		buildings.add(building);
	}

	/**
	 * Remove a building from the list of buildings in the city
	 *
	 * @param building
	 */
	@Override
	public void tearDown(IBuilding building) {
		building.destroy();
		buildings.remove(building);
	}

	@Override
	public IReputation getReputation(IPlayer player) {
		IReputation rep = reputation.get(player.getUuid());
		return rep;
	}

	@Override
	public void moveIn(IPlayer player, IReputation reputaion, Contributions contributions) {
		reputation.put(player.getUuid(), reputaion);
		playersContributions.put(player.getUuid(), contributions);
		if (this.equals(player.getHometown())) {
			citizen.add(player);
		}
	}

	/**
	 * {@inheritDoc}
	 * Update the contributions as the ware is moved
	 */
	@Override
	public int move(IWare ware, int amount, ICitizen player) {
		LOGGER.trace("Move {} of ware {} in {}", amount, ware.name(), getName());
		int moved = super.move(ware, amount, player);
		if (moved < 0) {
			if (getWare(ware).getAmount() == 0) {
				missingWares.put(ware, date.getCurrentDate());
			}
		} else if (moved > 0 && missingWares.containsKey(ware)) {
			missingWares.remove(ware);
		}
		Contributions contrib = null;
		if (player != null && player instanceof IPlayer) { // possible from test or from the city itself
			contrib = playersContributions.get(((IPlayer) player).getUuid());
			contrib.contribute(ware, moved);
		}
		return moved;
	}

	@Override
	public int getContribution(IPlayer player, IWare ware) {
		Contributions contribs = playersContributions.get(player.getUuid());
		if (contribs != null) {
			return contribs.getContribution(ware);
		} else {
			return 0;
		}
	}

	@Override
	public Map<IWare, LocalDateTime> getMissingWares() {
		return ImmutableMap.copyOf(missingWares);
	}

	@Override
	public List<IPlayer> getResidentPlayers() {
		ArrayList<IPlayer> players = new ArrayList<>();
		for (ICitizen iCitizen : citizen) {
			if (iCitizen instanceof IPlayer) {
				players.add((IPlayer) iCitizen);
			}
		}
		return players;
	}

	@Override
	public String getUniqueID() {
		return getName();
	}

	@Override
	public <T extends IBuilding> List<T> findBuilding(Class<T> buildingClass, Optional<IPlayer> owner) {
		if (owner.isPresent()) {
			return owner.get().findBuildings(this, buildingClass);
		}
		List<T> result = new ArrayList<>();
		for (IBuilding building : getBuildings()) {
			ClassChecker checker = new ClassChecker(building.getClass());
			if (checker.extendsClass(buildingClass) || checker.implementsInterface(buildingClass)) {
				if (building.getOwner().equals(owner)) {
					result.add((T) building);
				}
			}
		}
		return result;
	}

	@Override
	public boolean hasBuildingPermission(IPlayer player) {
		Boolean permission = buildingPermission.get(player.getUuid());
		return permission != null ? permission : false;
	}

	public void addBuildingPermission(IPlayer player) {
		buildingPermission.put(player.getUuid(), true);
	}

	@Override
	public double getPercentageRoad() {
		return roadTiles * 20 / internalPopulationCount();
	}

	@Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        City city = (City) o;

        return nameTemplate != null ? nameTemplate.equals(city.nameTemplate) : city.nameTemplate == null;

    }

    @Override
    public int hashCode() {
        return nameTemplate != null ? nameTemplate.hashCode() : 0;
    }
}